(*  Title:      Pure/Isar/expression.ML
    Author:     Clemens Ballarin, TU Muenchen

Locale expressions.
*)

signature EXPRESSION =
sig
  (* Locale expressions *)
  datatype 'term map = Positional of 'term option list | Named of (string * 'term) list
  type 'term expr = (string * ((string * bool) * 'term map)) list
  type expression_i = term expr * (binding * typ option * mixfix) list
  type expression = string expr * (binding * string option * mixfix) list

  (* Processing of context statements *)
  val cert_statement: Element.context_i list -> (term * term list) list list ->
    Proof.context -> (term * term list) list list * Proof.context
  val read_statement: Element.context list -> (string * string list) list list ->
    Proof.context -> (term * term list) list list * Proof.context

  (* Declaring locales *)
  val cert_declaration: expression_i -> (Proof.context -> Proof.context) -> Element.context_i list ->
    Proof.context -> ((binding * typ option * mixfix) list * (string * morphism) list
      * Element.context_i list) * ((string * typ) list * Proof.context)
  val cert_read_declaration: expression_i -> (Proof.context -> Proof.context) -> Element.context list ->
    Proof.context -> ((binding * typ option * mixfix) list * (string * morphism) list
      * Element.context_i list) * ((string * typ) list * Proof.context)
      (*FIXME*)
  val read_declaration: expression -> (Proof.context -> Proof.context) -> Element.context list ->
    Proof.context -> ((binding * typ option * mixfix) list * (string * morphism) list
      * Element.context_i list) * ((string * typ) list * Proof.context)
  val add_locale: binding -> binding -> expression_i -> Element.context_i list ->
    theory -> string * local_theory
  val add_locale_cmd: binding -> binding -> expression -> Element.context list ->
    theory -> string * local_theory

  (* Interpretation *)
  val cert_goal_expression: expression_i -> Proof.context ->
    (term list list * (string * morphism) list * morphism) * Proof.context
  val read_goal_expression: expression -> Proof.context ->
    (term list list * (string * morphism) list * morphism) * Proof.context
  val sublocale: string -> expression_i -> theory -> Proof.state
  val sublocale_cmd: string -> expression -> theory -> Proof.state
  val interpretation: expression_i -> (Attrib.binding * term) list -> theory -> Proof.state
  val interpretation_cmd: expression -> (Attrib.binding * string) list -> theory -> Proof.state
  val interpret: expression_i -> bool -> Proof.state -> Proof.state
  val interpret_cmd: expression -> bool -> Proof.state -> Proof.state
end;

structure Expression : EXPRESSION =
struct

datatype ctxt = datatype Element.ctxt;


(*** Expressions ***)

datatype 'term map =
  Positional of 'term option list |
  Named of (string * 'term) list;

type 'term expr = (string * ((string * bool) * 'term map)) list;

type expression = string expr * (binding * string option * mixfix) list;
type expression_i = term expr * (binding * typ option * mixfix) list;


(** Internalise locale names in expr **)

fun intern thy instances =  map (apfst (Locale.intern thy)) instances;


(** Parameters of expression.

   Sanity check of instantiations and extraction of implicit parameters.
   The latter only occurs iff strict = false.
   Positional instantiations are extended to match full length of parameter list
   of instantiated locale. **)

fun parameters_of thy strict (expr, fixed) =
  let
    fun reject_dups message xs =
      let val dups = duplicates (op =) xs
      in
        if null dups then () else error (message ^ commas dups)
      end;

    fun match_bind (n, b) = (n = Binding.name_of b);
    fun parm_eq ((b1, mx1: mixfix), (b2, mx2)) =
      Binding.eq_name (b1, b2) andalso
        (mx1 = mx2 orelse
          error ("Conflicting syntax for parameter " ^ quote (Binding.str_of b1) ^ " in expression"));

    fun params_loc loc =
      (Locale.params_of thy loc |> map (fn (p, _, mx) => (p, mx)), loc);
    fun params_inst (expr as (loc, (prfx, Positional insts))) =
          let
            val (ps, loc') = params_loc loc;
            val d = length ps - length insts;
            val insts' =
              if d < 0 then error ("More arguments than parameters in instantiation of locale " ^
                quote (Locale.extern thy loc))
              else insts @ replicate d NONE;
            val ps' = (ps ~~ insts') |>
              map_filter (fn (p, NONE) => SOME p | (_, SOME _) => NONE);
          in (ps', (loc', (prfx, Positional insts'))) end
      | params_inst (expr as (loc, (prfx, Named insts))) =
          let
            val _ = reject_dups "Duplicate instantiation of the following parameter(s): "
              (map fst insts);

            val (ps, loc') = params_loc loc;
            val ps' = fold (fn (p, _) => fn ps =>
              if AList.defined match_bind ps p then AList.delete match_bind p ps
              else error (quote p ^" not a parameter of instantiated expression.")) insts ps;
          in (ps', (loc', (prfx, Named insts))) end;
    fun params_expr is =
          let
            val (is', ps') = fold_map (fn i => fn ps =>
              let
                val (ps', i') = params_inst i;
                val ps'' = distinct parm_eq (ps @ ps');
              in (i', ps'') end) is []
          in (ps', is') end;

    val (implicit, expr') = params_expr expr;

    val implicit' = map (#1 #> Name.of_binding) implicit;
    val fixed' = map (#1 #> Name.of_binding) fixed;
    val _ = reject_dups "Duplicate fixed parameter(s): " fixed';
    val implicit'' =
      if strict then []
      else
        let val _ = reject_dups
          "Parameter(s) declared simultaneously in expression and for clause: " (implicit' @ fixed')
        in map (fn (b, mx) => (b, NONE, mx)) implicit end;

  in (expr', implicit'' @ fixed) end;


(** Read instantiation **)

(* Parse positional or named instantiation *)

local

fun prep_inst prep_term ctxt parms (Positional insts) =
      (insts ~~ parms) |> map (fn
        (NONE, p) => Free (p, dummyT) |
        (SOME t, _) => prep_term ctxt t)
  | prep_inst prep_term ctxt parms (Named insts) =
      parms |> map (fn p => case AList.lookup (op =) insts p of
        SOME t => prep_term ctxt t |
        NONE => Free (p, dummyT));

in

fun parse_inst x = prep_inst Syntax.parse_term x;
fun make_inst x = prep_inst (K I) x;

end;


(* Instantiation morphism *)

fun inst_morph (parm_names, parm_types) ((prfx, strict), insts') ctxt =
  let
    (* parameters *)
    val type_parm_names = fold Term.add_tfreesT parm_types [] |> map fst;

    (* type inference and contexts *)
    val parm_types' = map (TypeInfer.paramify_vars o Logic.varifyT) parm_types;
    val type_parms = fold Term.add_tvarsT parm_types' [] |> map (Logic.mk_type o TVar);
    val arg = type_parms @ map2 TypeInfer.constrain parm_types' insts';
    val res = Syntax.check_terms ctxt arg;
    val ctxt' = ctxt |> fold Variable.auto_fixes res;

    (* instantiation *)
    val (type_parms'', res') = chop (length type_parms) res;
    val insts'' = (parm_names ~~ res') |> map_filter
      (fn (inst as (x, Free (y, _))) => if x = y then NONE else SOME inst |
        inst => SOME inst);
    val instT = Symtab.make (type_parm_names ~~ map Logic.dest_type type_parms'');
    val inst = Symtab.make insts'';
  in
    (Element.inst_morphism (ProofContext.theory_of ctxt) (instT, inst) $>
      Morphism.binding_morphism (Binding.prefix strict prfx), ctxt')
  end;


(*** Locale processing ***)

(** Parsing **)

fun parse_elem prep_typ prep_term ctxt =
  Element.map_ctxt
   {binding = I,
    typ = prep_typ ctxt,
    term = prep_term (ProofContext.set_mode ProofContext.mode_schematic ctxt),
    pattern = prep_term (ProofContext.set_mode ProofContext.mode_pattern ctxt),
    fact = I,
    attrib = I};

fun parse_concl prep_term ctxt concl =
  (map o map) (fn (t, ps) =>
    (prep_term (ProofContext.set_mode ProofContext.mode_schematic ctxt) t, (* FIXME ?? *)
      map (prep_term (ProofContext.set_mode ProofContext.mode_pattern ctxt)) ps)) concl;


(** Simultaneous type inference: instantiations + elements + conclusion **)

local

fun mk_type T = (Logic.mk_type T, []);
fun mk_term t = (t, []);
fun mk_propp (p, pats) = (Syntax.type_constraint propT p, pats);

fun dest_type (T, []) = Logic.dest_type T;
fun dest_term (t, []) = t;
fun dest_propp (p, pats) = (p, pats);

fun extract_inst (_, (_, ts)) = map mk_term ts;
fun restore_inst ((l, (p, _)), cs) = (l, (p, map dest_term cs));

fun extract_elem (Fixes fixes) = map (#2 #> the_list #> map mk_type) fixes
  | extract_elem (Constrains csts) = map (#2 #> single #> map mk_type) csts
  | extract_elem (Assumes asms) = map (#2 #> map mk_propp) asms
  | extract_elem (Defines defs) = map (fn (_, (t, ps)) => [mk_propp (t, ps)]) defs
  | extract_elem (Notes _) = [];

fun restore_elem (Fixes fixes, css) =
      (fixes ~~ css) |> map (fn ((x, _, mx), cs) =>
        (x, cs |> map dest_type |> try hd, mx)) |> Fixes
  | restore_elem (Constrains csts, css) =
      (csts ~~ css) |> map (fn ((x, _), cs) =>
        (x, cs |> map dest_type |> hd)) |> Constrains
  | restore_elem (Assumes asms, css) =
      (asms ~~ css) |> map (fn ((b, _), cs) => (b, map dest_propp cs)) |> Assumes
  | restore_elem (Defines defs, css) =
      (defs ~~ css) |> map (fn ((b, _), [c]) => (b, dest_propp c)) |> Defines
  | restore_elem (Notes notes, _) = Notes notes;

fun check cs context =
  let
    fun prep (_, pats) (ctxt, t :: ts) =
      let val ctxt' = Variable.auto_fixes t ctxt
      in
        ((t, Syntax.check_props (ProofContext.set_mode ProofContext.mode_pattern ctxt') pats),
          (ctxt', ts))
      end
    val (cs', (context', _)) = fold_map prep cs
      (context, Syntax.check_terms
        (ProofContext.set_mode ProofContext.mode_schematic context) (map fst cs));
  in (cs', context') end;

in

fun check_autofix insts elems concl ctxt =
  let
    val inst_cs = map extract_inst insts;
    val elem_css = map extract_elem elems;
    val concl_cs = (map o map) mk_propp concl;
    (* Type inference *)
    val (inst_cs' :: css', ctxt') =
      (fold_burrow o fold_burrow) check (inst_cs :: elem_css @ [concl_cs]) ctxt;
    val (elem_css', [concl_cs']) = chop (length elem_css) css';
  in
    (map restore_inst (insts ~~ inst_cs'), map restore_elem (elems ~~ elem_css'),
      concl_cs', ctxt')
  end;

end;


(** Prepare locale elements **)

fun declare_elem prep_vars (Fixes fixes) ctxt =
      let val (vars, _) = prep_vars fixes ctxt
      in ctxt |> ProofContext.add_fixes_i vars |> snd end
  | declare_elem prep_vars (Constrains csts) ctxt =
      ctxt |> prep_vars (map (fn (x, T) => (Binding.name x, SOME T, NoSyn)) csts) |> snd
  | declare_elem _ (Assumes _) ctxt = ctxt
  | declare_elem _ (Defines _) ctxt = ctxt
  | declare_elem _ (Notes _) ctxt = ctxt;

(** Finish locale elements **)

fun closeup _ _ false elem = elem
  | closeup ctxt parms true elem =
      let
        fun close_frees t =
          let
            val rev_frees =
              Term.fold_aterms (fn Free (x, T) =>
                if AList.defined (op =) parms x then I else insert (op =) (x, T) | _ => I) t [];
          in Term.list_all_free (rev rev_frees, t) end; (* FIXME use fold Logic.all *)
  (* FIXME consider closing in syntactic phase *)

        fun no_binds [] = []
          | no_binds _ = error "Illegal term bindings in context element";
      in
        (case elem of
          Assumes asms => Assumes (asms |> map (fn (a, propps) =>
            (a, map (fn (t, ps) => (close_frees t, no_binds ps)) propps)))
        | Defines defs => Defines (defs |> map (fn ((name, atts), (t, ps)) =>
            let val ((c, _), t') = LocalDefs.cert_def ctxt (close_frees t)
            in ((Thm.def_binding_optional (Binding.name c) name, atts), (t', no_binds ps)) end))
        | e => e)
      end;

fun finish_primitive parms _ (Fixes fixes) = Fixes (map (fn (binding, _, mx) =>
      let val x = Binding.name_of binding
      in (binding, AList.lookup (op =) parms x, mx) end) fixes)
  | finish_primitive _ _ (Constrains _) = Constrains []
  | finish_primitive _ close (Assumes asms) = close (Assumes asms)
  | finish_primitive _ close (Defines defs) = close (Defines defs)
  | finish_primitive _ _ (Notes facts) = Notes facts;

fun finish_inst ctxt parms do_close (loc, (prfx, inst)) =
  let
    val thy = ProofContext.theory_of ctxt;
    val (parm_names, parm_types) = Locale.params_of thy loc |>
      map_split (fn (b, SOME T, _) => (Binding.name_of b, T));
    val (morph, _) = inst_morph (parm_names, parm_types) (prfx, inst) ctxt;
  in (loc, morph) end;

fun finish_elem ctxt parms do_close elem =
  let
    val elem' = finish_primitive parms (closeup ctxt parms do_close) elem;
  in elem' end

fun finish ctxt parms do_close insts elems =
  let
    val deps = map (finish_inst ctxt parms do_close) insts;
    val elems' = map (finish_elem ctxt parms do_close) elems;
  in (deps, elems') end;


(** Process full context statement: instantiations + elements + conclusion **)

(* Interleave incremental parsing and type inference over entire parsed stretch. *)

local

fun prep_full_context_statement parse_typ parse_prop prep_vars_elem prep_inst prep_vars_inst prep_expr
    strict do_close raw_import init_body raw_elems raw_concl ctxt1 =
  let
    val thy = ProofContext.theory_of ctxt1;

    val (raw_insts, fixed) = parameters_of thy strict (apfst (prep_expr thy) raw_import);

    fun prep_insts (loc, (prfx, inst)) (i, insts, ctxt) =
      let
        val (parm_names, parm_types) = Locale.params_of thy loc |>
          map_split (fn (b, SOME T, _) => (Name.of_binding b, T))
            (*FIXME return value of Locale.params_of has strange type*)
        val inst' = prep_inst ctxt parm_names inst;
        val parm_types' = map (TypeInfer.paramify_vars o
          Term.map_type_tvar (fn ((x, _), S) => TVar ((x, i), S)) o Logic.varifyT) parm_types;
        val inst'' = map2 TypeInfer.constrain parm_types' inst';
        val insts' = insts @ [(loc, (prfx, inst''))];
        val (insts'', _, _, ctxt' (* FIXME not used *) ) = check_autofix insts' [] [] ctxt;
        val inst''' = insts'' |> List.last |> snd |> snd;
        val (morph, _) = inst_morph (parm_names, parm_types) (prfx, inst''') ctxt;
        val ctxt'' = Locale.activate_declarations thy (loc, morph) ctxt;
      in (i+1, insts', ctxt'') end;

    fun prep_elem insts raw_elem (elems, ctxt) =
      let
        val ctxt' = declare_elem prep_vars_elem raw_elem ctxt;
        val elems' = elems @ [parse_elem parse_typ parse_prop ctxt' raw_elem];
        val (_, _, _, ctxt'') = check_autofix insts elems' [] ctxt';
      in (elems', ctxt') end;

    fun prep_concl raw_concl (insts, elems, ctxt) =
      let
        val concl = parse_concl parse_prop ctxt raw_concl;
      in check_autofix insts elems concl ctxt end;

    val fors = prep_vars_inst fixed ctxt1 |> fst;
    val ctxt2 = ctxt1 |> ProofContext.add_fixes_i fors |> snd;
    val (_, insts', ctxt3) = fold prep_insts raw_insts (0, [], ctxt2);
    val ctxt4 = init_body ctxt3;
    val (elems, ctxt5) = fold (prep_elem insts') raw_elems ([], ctxt4);
    val (insts, elems', concl, ctxt6) =
      prep_concl raw_concl (insts', elems, ctxt5);

    (* Retrieve parameter types *)
    val xs = fold (fn Fixes fixes => (fn ps => ps @ map (Name.of_binding o #1) fixes)
      | _ => fn ps => ps) (Fixes fors :: elems') [];
    val (Ts, ctxt7) = fold_map ProofContext.inferred_param xs ctxt6;
    val parms = xs ~~ Ts;  (* params from expression and elements *)

    val Fixes fors' = finish_primitive parms I (Fixes fors);
    val (deps, elems'') = finish ctxt6 parms do_close insts elems';

  in ((fors', deps, elems'', concl), (parms, ctxt7)) end

in

fun cert_full_context_statement x =
  prep_full_context_statement (K I) (K I) ProofContext.cert_vars
  make_inst ProofContext.cert_vars (K I) x;
fun cert_read_full_context_statement x =
  prep_full_context_statement Syntax.parse_typ Syntax.parse_prop ProofContext.read_vars
  make_inst ProofContext.cert_vars (K I) x;
fun read_full_context_statement x =
  prep_full_context_statement Syntax.parse_typ Syntax.parse_prop ProofContext.read_vars
  parse_inst ProofContext.read_vars intern x;

end;


(* Context statement: elements + conclusion *)

local

fun prep_statement prep activate raw_elems raw_concl context =
  let
     val ((_, _, elems, concl), _) =
       prep true false ([], []) I raw_elems raw_concl context;
     val (_, context') = context |>
       ProofContext.set_stmt true |>
       activate elems;
  in (concl, context') end;

in

fun cert_statement x = prep_statement cert_full_context_statement Element.activate_i x;
fun read_statement x = prep_statement read_full_context_statement Element.activate x;

end;


(* Locale declaration: import + elements *)

local

fun prep_declaration prep activate raw_import init_body raw_elems context =
  let
    val ((fixed, deps, elems, _), (parms, ctxt')) =
      prep false true raw_import init_body raw_elems [] context ;
    (* Declare parameters and imported facts *)
    val context' = context |>
      ProofContext.add_fixes_i fixed |> snd |>
      fold Locale.activate_local_facts deps;
    val (elems', _) = context' |>
      ProofContext.set_stmt true |>
      activate elems;
  in ((fixed, deps, elems'), (parms, ctxt')) end;

in

fun cert_declaration x = prep_declaration cert_full_context_statement Element.activate_i x;
fun cert_read_declaration x = prep_declaration cert_read_full_context_statement Element.activate x;
fun read_declaration x = prep_declaration read_full_context_statement Element.activate x;

end;


(* Locale expression to set up a goal *)

local

fun props_of thy (name, morph) =
  let
    val (asm, defs) = Locale.specification_of thy name;
  in
    (case asm of NONE => defs | SOME asm => asm :: defs) |> map (Morphism.term morph)
  end;

fun prep_goal_expression prep expression context =
  let
    val thy = ProofContext.theory_of context;

    val ((fixed, deps, _, _), _) =
      prep true true expression I [] [] context;
    (* proof obligations *)
    val propss = map (props_of thy) deps;

    val goal_ctxt = context |>
      ProofContext.add_fixes_i fixed |> snd |>
      (fold o fold) Variable.auto_fixes propss;

    val export = Variable.export_morphism goal_ctxt context;
    val exp_fact = Drule.zero_var_indexes_list o map Thm.strip_shyps o Morphism.fact export;
    val exp_term = TermSubst.zero_var_indexes o Morphism.term export;
    val exp_typ = Logic.type_map exp_term;
    val export' = Morphism.morphism {binding = I, typ = exp_typ, term = exp_term, fact = exp_fact};
  in ((propss, deps, export'), goal_ctxt) end;

in

fun cert_goal_expression x = prep_goal_expression cert_full_context_statement x;
fun read_goal_expression x = prep_goal_expression read_full_context_statement x;

end;


(*** Locale declarations ***)

(* extract specification text *)

val norm_term = Envir.beta_norm oo Term.subst_atomic;

fun bind_def ctxt eq (xs, env, eqs) =
  let
    val _ = LocalDefs.cert_def ctxt eq;
    val ((y, T), b) = LocalDefs.abs_def eq;
    val b' = norm_term env b;
    fun err msg = error (msg ^ ": " ^ quote y);
  in
    case filter (fn (Free (y', _), _) => y = y' | _ => false) env of
      [] => (Term.add_frees b' xs, (Free (y, T), b') :: env, eq :: eqs) |
      dups => if forall (fn (_, b'') => b' aconv b'') dups
        then (xs, env, eqs)
        else err "Attempt to redefine variable"
  end;

(* text has the following structure:
       (((exts, exts'), (ints, ints')), (xs, env, defs))
   where
     exts: external assumptions (terms in assumes elements)
     exts': dito, normalised wrt. env
     ints: internal assumptions (terms in assumptions from insts)
     ints': dito, normalised wrt. env
     xs: the free variables in exts' and ints' and rhss of definitions,
       this includes parameters except defined parameters
     env: list of term pairs encoding substitutions, where the first term
       is a free variable; substitutions represent defines elements and
       the rhs is normalised wrt. the previous env
     defs: the equations from the defines elements
   *)

fun eval_text _ _ (Fixes _) text = text
  | eval_text _ _ (Constrains _) text = text
  | eval_text _ is_ext (Assumes asms)
        (((exts, exts'), (ints, ints')), (xs, env, defs)) =
      let
        val ts = maps (map #1 o #2) asms;
        val ts' = map (norm_term env) ts;
        val spec' =
          if is_ext then ((exts @ ts, exts' @ ts'), (ints, ints'))
          else ((exts, exts'), (ints @ ts, ints' @ ts'));
      in (spec', (fold Term.add_frees ts' xs, env, defs)) end
  | eval_text ctxt _ (Defines defs) (spec, binds) =
      (spec, fold (bind_def ctxt o #1 o #2) defs binds)
  | eval_text _ _ (Notes _) text = text;

fun eval_inst ctxt (loc, morph) text =
  let
    val thy = ProofContext.theory_of ctxt;
    val (asm, defs) = Locale.specification_of thy loc;
    val asm' = Option.map (Morphism.term morph) asm;
    val defs' = map (Morphism.term morph) defs;
    val text' = text |>
      (if is_some asm
        then eval_text ctxt false (Assumes [(Attrib.empty_binding, [(the asm', [])])])
        else I) |>
      (if not (null defs)
        then eval_text ctxt false (Defines (map (fn def => (Attrib.empty_binding, (def, []))) defs'))
        else I)
(* FIXME clone from locale.ML *)
  in text' end;

fun eval_elem ctxt elem text =
  let
    val text' = eval_text ctxt true elem text;
  in text' end;

fun eval ctxt deps elems =
  let
    val text' = fold (eval_inst ctxt) deps ((([], []), ([], [])), ([], [], []));
    val ((spec, (_, _, defs))) = fold (eval_elem ctxt) elems text';
  in (spec, defs) end;

(* axiomsN: name of theorem set with destruct rules for locale predicates,
     also name suffix of delta predicates and assumptions. *)

val axiomsN = "axioms";

local

(* introN: name of theorems for introduction rules of locale and
     delta predicates *)

val introN = "intro";

fun atomize_spec thy ts =
  let
    val t = Logic.mk_conjunction_balanced ts;
    val body = ObjectLogic.atomize_term thy t;
    val bodyT = Term.fastype_of body;
  in
    if bodyT = propT then (t, propT, Thm.reflexive (Thm.cterm_of thy t))
    else (body, bodyT, ObjectLogic.atomize (Thm.cterm_of thy t))
  end;

(* achieve plain syntax for locale predicates (without "PROP") *)

fun aprop_tr' n c = (Syntax.constN ^ c, fn ctxt => fn args =>
  if length args = n then
    Syntax.const "_aprop" $
      Term.list_comb (Syntax.free (Consts.extern (ProofContext.consts_of ctxt) c), args)
  else raise Match);

(* define one predicate including its intro rule and axioms
   - bname: predicate name
   - parms: locale parameters
   - defs: thms representing substitutions from defines elements
   - ts: terms representing locale assumptions (not normalised wrt. defs)
   - norm_ts: terms representing locale assumptions (normalised wrt. defs)
   - thy: the theory
*)

fun def_pred bname parms defs ts norm_ts thy =
  let
    val name = Sign.full_name thy bname;

    val (body, bodyT, body_eq) = atomize_spec thy norm_ts;
    val env = Term.add_free_names body [];
    val xs = filter (member (op =) env o #1) parms;
    val Ts = map #2 xs;
    val extraTs =
      (Term.add_tfrees body [] \\ fold Term.add_tfreesT Ts [])
      |> Library.sort_wrt #1 |> map TFree;
    val predT = map Term.itselfT extraTs ---> Ts ---> bodyT;

    val args = map Logic.mk_type extraTs @ map Free xs;
    val head = Term.list_comb (Const (name, predT), args);
    val statement = ObjectLogic.ensure_propT thy head;

    val ([pred_def], defs_thy) =
      thy
      |> bodyT = propT ? Sign.add_advanced_trfuns ([], [], [aprop_tr' (length args) name], [])
      |> Sign.declare_const [] ((bname, predT), NoSyn) |> snd
      |> PureThy.add_defs false
        [((Binding.map_name Thm.def_name bname, Logic.mk_equals (head, body)), [Thm.kind_internal])];
    val defs_ctxt = ProofContext.init defs_thy |> Variable.declare_term head;

    val cert = Thm.cterm_of defs_thy;

    val intro = Goal.prove_global defs_thy [] norm_ts statement (fn _ =>
      MetaSimplifier.rewrite_goals_tac [pred_def] THEN
      Tactic.compose_tac (false, body_eq RS Drule.equal_elim_rule1, 1) 1 THEN
      Tactic.compose_tac (false,
        Conjunction.intr_balanced (map (Thm.assume o cert) norm_ts), 0) 1);

    val conjuncts =
      (Drule.equal_elim_rule2 OF [body_eq,
        MetaSimplifier.rewrite_rule [pred_def] (Thm.assume (cert statement))])
      |> Conjunction.elim_balanced (length ts);
    val axioms = ts ~~ conjuncts |> map (fn (t, ax) =>
      Element.prove_witness defs_ctxt t
       (MetaSimplifier.rewrite_goals_tac defs THEN
        Tactic.compose_tac (false, ax, 0) 1));
  in ((statement, intro, axioms), defs_thy) end;

in

(* main predicate definition function *)

fun define_preds pname parms (((exts, exts'), (ints, ints')), defs) thy =
  let
    val defs' = map (cterm_of thy #> Assumption.assume #> Drule.gen_all #> Drule.abs_def) defs;

    val (a_pred, a_intro, a_axioms, thy'') =
      if null exts then (NONE, NONE, [], thy)
      else
        let
          val aname = if null ints then pname else Binding.suffix_name ("_" ^ axiomsN) pname
          val ((statement, intro, axioms), thy') =
            thy
            |> def_pred aname parms defs' exts exts';
          val (_, thy'') =
            thy'
            |> Sign.mandatory_path (Binding.name_of aname)
            |> PureThy.note_thmss Thm.internalK
              [((Binding.name introN, []), [([intro], [Locale.unfold_attrib])])]
            ||> Sign.restore_naming thy';
          in (SOME statement, SOME intro, axioms, thy'') end;
    val (b_pred, b_intro, b_axioms, thy'''') =
      if null ints then (NONE, NONE, [], thy'')
      else
        let
          val ((statement, intro, axioms), thy''') =
            thy''
            |> def_pred pname parms defs' (ints @ the_list a_pred) (ints' @ the_list a_pred);
          val (_, thy'''') =
            thy'''
            |> Sign.mandatory_path (Binding.name_of pname)
            |> PureThy.note_thmss Thm.internalK
                 [((Binding.name introN, []), [([intro], [Locale.intro_attrib])]),
                  ((Binding.name axiomsN, []),
                    [(map (Drule.standard o Element.conclude_witness) axioms, [])])]
            ||> Sign.restore_naming thy''';
        in (SOME statement, SOME intro, axioms, thy'''') end;
  in ((a_pred, a_intro, a_axioms), (b_pred, b_intro, b_axioms), thy'''') end;

end;


local

fun assumes_to_notes (Assumes asms) axms =
      fold_map (fn (a, spec) => fn axs =>
          let val (ps, qs) = chop (length spec) axs
          in ((a, [(ps, [])]), qs) end) asms axms
      |> apfst (curry Notes Thm.assumptionK)
  | assumes_to_notes e axms = (e, axms);

fun defines_to_notes thy (Defines defs) =
      Notes (Thm.definitionK, map (fn (a, (def, _)) =>
        (a, [([Assumption.assume (cterm_of thy def)],
          [(Attrib.internal o K) Locale.witness_attrib])])) defs)
  | defines_to_notes _ e = e;

fun gen_add_locale prep_decl
    bname raw_predicate_bname raw_import raw_body thy =
  let
    val name = Sign.full_name thy bname;
    val _ = Locale.defined thy name andalso
      error ("Duplicate definition of locale " ^ quote name);

    val ((fixed, deps, body_elems), (parms, ctxt')) =
      prep_decl raw_import I raw_body (ProofContext.init thy);
    val text as (((_, exts'), _), defs) = eval ctxt' deps body_elems;

    val predicate_bname =
      if Binding.is_empty raw_predicate_bname then bname
      else raw_predicate_bname;
    val ((a_statement, a_intro, a_axioms), (b_statement, b_intro, b_axioms), thy') =
      define_preds predicate_bname parms text thy;

    val extraTs = fold Term.add_tfrees exts' [] \\ fold Term.add_tfreesT (map snd parms) [];
    val _ =
      if null extraTs then ()
      else warning ("Additional type variable(s) in locale specification " ^ quote (Binding.str_of bname));

    val a_satisfy = Element.satisfy_morphism a_axioms;
    val b_satisfy = Element.satisfy_morphism b_axioms;

    val params = fixed @
      (body_elems |> map_filter (fn Fixes fixes => SOME fixes | _ => NONE) |> flat);
    val asm = if is_some b_statement then b_statement else a_statement;

    val notes =
        if is_some asm
        then [(Thm.internalK, [((Binding.suffix_name ("_" ^ axiomsN) bname, []),
          [([Assumption.assume (cterm_of thy' (the asm))],
            [(Attrib.internal o K) Locale.witness_attrib])])])]
        else [];

    val notes' = body_elems |>
      map (defines_to_notes thy') |>
      map (Element.morph_ctxt a_satisfy) |>
      (fn elems => fold_map assumes_to_notes elems (map Element.conclude_witness a_axioms)) |>
      fst |>
      map (Element.morph_ctxt b_satisfy) |>
      map_filter (fn Notes notes => SOME notes | _ => NONE);

    val deps' = map (fn (l, morph) => (l, morph $> b_satisfy)) deps;
    val axioms = map Element.conclude_witness b_axioms;

    val loc_ctxt = thy'
      |> Locale.register_locale bname (extraTs, params)
          (asm, rev defs) (a_intro, b_intro) axioms ([], [])
          (map (fn n => (n, stamp ())) notes |> rev) (map (fn d => (d, stamp ())) deps' |> rev)
      |> TheoryTarget.init (SOME name)
      |> fold (fn (kind, facts) => LocalTheory.notes kind facts #> snd) notes';

  in (name, loc_ctxt) end;

in

val add_locale = gen_add_locale cert_declaration;
val add_locale_cmd = gen_add_locale read_declaration;

end;


(*** Interpretation ***)

(** Interpretation between locales: declaring sublocale relationships **)

local

fun gen_sublocale prep_expr intern raw_target expression thy =
  let
    val target = intern thy raw_target;
    val target_ctxt = Locale.init target thy;

    val ((propss, deps, export), goal_ctxt) = prep_expr expression target_ctxt;

    fun after_qed witss = ProofContext.theory (
      fold2 (fn (name, morph) => fn wits => Locale.add_dependency target
        (name, morph $> Element.satisfy_morphism wits $> export)) deps witss #>
      (fn thy => fold_rev Locale.activate_global_facts
          (Locale.get_registrations thy) thy));
  in Element.witness_proof after_qed propss goal_ctxt end;

in

fun sublocale x = gen_sublocale cert_goal_expression (K I) x;
fun sublocale_cmd x = gen_sublocale read_goal_expression Locale.intern x;

end;


(** Interpretation in theories **)

local

fun gen_interpretation prep_expr parse_prop prep_attr
    expression equations theory =
  let
    val ((propss, regs, export), expr_ctxt) = ProofContext.init theory
      |> prep_expr expression;

    val eqns = map (parse_prop expr_ctxt o snd) equations |> Syntax.check_terms expr_ctxt;
    val eq_attns = map ((apsnd o map) (prep_attr theory) o fst) equations;
    val goal_ctxt = fold Variable.auto_fixes eqns expr_ctxt;
    val export' = Variable.export_morphism goal_ctxt expr_ctxt;

    fun store_reg ((name, morph), wits) thy =
      let
        val wits' = map (Element.morph_witness export') wits;
        val morph' = morph $> Element.satisfy_morphism wits';
      in
        thy
        |> Locale.add_registration (name, (morph', export))
        |> pair (name, morph')
      end;

    fun store_eqns_activate regs [] thy =
          thy
          |> fold (fn (name, morph) =>
               Locale.activate_global_facts (name, morph $> export)) regs
      | store_eqns_activate regs eqs thy =
          let
            val eqs' = eqs |> map (Morphism.thm (export' $> export) #>
              LocalDefs.meta_rewrite_rule (ProofContext.init thy) #>
              Drule.abs_def);
            val eq_morph = Element.eq_morphism thy eqs';
            val eq_attns' = map ((apsnd o map) (Attrib.attribute_i thy)) eq_attns;
          in
            thy
            |> fold (fn (name, morph) =>
                Locale.amend_registration eq_morph (name, morph) #>
                Locale.activate_global_facts (name, morph $> eq_morph $> export)) regs
            |> PureThy.note_thmss Thm.lemmaK (eq_attns' ~~ map (fn eq => [([eq], [])]) eqs')
            |> snd
          end;

    fun after_qed wits eqs = ProofContext.theory (fold_map store_reg (regs ~~ wits)
        #-> (fn regs => store_eqns_activate regs eqs));

  in Element.witness_proof_eqs after_qed propss eqns goal_ctxt end;

in

fun interpretation x = gen_interpretation cert_goal_expression (K I) (K I) x;
fun interpretation_cmd x = gen_interpretation read_goal_expression
  Syntax.parse_prop Attrib.intern_src x;

end;


(** Interpretation in proof contexts **)

local

fun gen_interpret prep_expr
    expression int state =
  let
    val _ = Proof.assert_forward_or_chain state;
    val ctxt = Proof.context_of state;

    val ((propss, regs, export), goal_ctxt) = prep_expr expression ctxt;

    fun store_reg (name, morph) thms =
      let
        val morph' = morph $> Element.satisfy_morphism thms $> export;
      in Locale.activate_local_facts (name, morph') end;

    fun after_qed wits =
      Proof.map_context (fold2 store_reg regs wits);
  in
    state
    |> Element.witness_local_proof after_qed "interpret" propss goal_ctxt int
  end;

in

fun interpret x = gen_interpret cert_goal_expression x;
fun interpret_cmd x = gen_interpret read_goal_expression x;

end;

end;

