(*  Title:      Pure/PIDE/command.ML
    Author:     Makarius

Prover command execution: read -- eval -- print.
*)

signature COMMAND =
sig
  type span
  type eval_process
  type eval = {exec_id: Document_ID.exec, eval_process: eval_process}
  type print_process
  type print = {name: string, pri: int, exec_id: Document_ID.exec, print_process: print_process}
  type exec = eval * print list
  val read: (unit -> theory) -> span -> Toplevel.transition
  val no_eval: eval
  val eval_result_state: eval -> Toplevel.state
  val eval: (unit -> theory) -> span -> eval -> eval
  val print: string -> eval -> print list
  type print_fn = Toplevel.transition -> Toplevel.state -> unit
  val print_function: {name: string, pri: int} -> (string -> print_fn option) -> unit
  val no_exec: exec
  val exec_ids: exec -> Document_ID.exec list
  val exec_run: exec -> unit
  val stable_eval: eval -> bool
  val stable_print: print -> bool
end;

structure Command: COMMAND =
struct

(** memo results -- including physical interrupts! **)

datatype 'a expr =
  Expr of unit -> 'a |
  Result of 'a Exn.result;

abstype 'a memo = Memo of 'a expr Synchronized.var
with

fun memo e = Memo (Synchronized.var "Command.memo" (Expr e));
fun memo_value a = Memo (Synchronized.var "Command.memo" (Result (Exn.Res a)));

fun memo_eval (Memo v) =
  (case Synchronized.value v of
    Result res => res
  | _ =>
      Synchronized.guarded_access v
        (fn Result res => SOME (res, Result res)
          | Expr e =>
              let val res = Exn.capture e ();  (*sic!*)
              in SOME (res, Result res) end))
  |> Exn.release;

fun memo_fork params (Memo v) =
  (case Synchronized.value v of
    Result _ => ()
  | _ => ignore ((singleton o Future.forks) params (fn () => memo_eval (Memo v))));

fun memo_result (Memo v) =
  (case Synchronized.value v of
    Result res => Exn.release res
  | _ => raise Fail "Unfinished memo result");

fun memo_stable (Memo v) =
  (case Synchronized.value v of
    Expr _ => true
  | Result res => not (Exn.is_interrupt_exn res));

end;



(** main phases **)

type span = Token.T list

type eval_state =
  {failed: bool, malformed: bool, command: Toplevel.transition, state: Toplevel.state};
type eval_process = eval_state memo;
type eval = {exec_id: Document_ID.exec, eval_process: eval_process};

type print_process = unit memo;
type print = {name: string, pri: int, exec_id: Document_ID.exec, print_process: print_process};


(* read *)

fun read init span =
  let
    val outer_syntax = #2 (Outer_Syntax.get_syntax ());
    val command_reports = Outer_Syntax.command_reports outer_syntax;

    val proper_range =
      Position.set_range (Token.position_range_of (#1 (take_suffix Token.is_improper span)));
    val pos =
      (case find_first Token.is_command span of
        SOME tok => Token.position_of tok
      | NONE => proper_range);

    val (is_malformed, token_reports) = Thy_Syntax.reports_of_tokens span;
    val _ = Position.reports_text (token_reports @ maps command_reports span);
  in
    if is_malformed then Toplevel.malformed pos "Malformed command syntax"
    else
      (case Outer_Syntax.read_spans outer_syntax span of
        [tr] =>
          if Keyword.is_control (Toplevel.name_of tr) then
            Toplevel.malformed pos "Illegal control command"
          else Toplevel.modify_init init tr
      | [] => Toplevel.ignored (Position.set_range (Token.position_range_of span))
      | _ => Toplevel.malformed proper_range "Exactly one command expected")
      handle ERROR msg => Toplevel.malformed proper_range msg
  end;


(* eval *)

val no_eval_state: eval_state =
  {failed = false, malformed = false, command = Toplevel.empty, state = Toplevel.toplevel};

val no_eval: eval = {exec_id = Document_ID.none, eval_process = memo_value no_eval_state};

fun eval_result ({eval_process, ...}: eval) = memo_result eval_process;
val eval_result_state = #state o eval_result;

local

fun run int tr st =
  if Goal.future_enabled () andalso Keyword.is_diag (Toplevel.name_of tr) then
    (Goal.fork_params {name = "Toplevel.diag", pos = Toplevel.pos_of tr, pri = ~1}
      (fn () => Toplevel.command_exception int tr st); ([], SOME st))
  else Toplevel.command_errors int tr st;

fun check_cmts span tr st' =
  Toplevel.setmp_thread_position tr
    (fn () =>
      Outer_Syntax.side_comments span |> maps (fn cmt =>
        (Thy_Output.check_text (Token.source_position_of cmt) st'; [])
          handle exn => ML_Compiler.exn_messages_ids exn)) ();

fun proof_status tr st =
  (case try Toplevel.proof_of st of
    SOME prf => Toplevel.status tr (Proof.status_markup prf)
  | NONE => ());

fun eval_state span tr ({malformed, state = st, ...}: eval_state) =
  if malformed then
    {failed = true, malformed = malformed, command = tr, state = Toplevel.toplevel}
  else
    let
      val malformed' = Toplevel.is_malformed tr;
      val is_init = Toplevel.is_init tr;
      val is_proof = Keyword.is_proof (Toplevel.name_of tr);

      val _ = Multithreading.interrupted ();
      val _ = Toplevel.status tr Markup.running;
      val (errs1, result) = run (is_init orelse is_proof) (Toplevel.set_print false tr) st;
      val errs2 = (case result of NONE => [] | SOME st' => check_cmts span tr st');
      val errs = errs1 @ errs2;
      val _ = Toplevel.status tr Markup.finished;
      val _ = List.app (Future.error_msg (Toplevel.pos_of tr)) errs;
    in
      (case result of
        NONE =>
          let
            val _ = if null errs then Exn.interrupt () else ();
            val _ = Toplevel.status tr Markup.failed;
          in {failed = true, malformed = malformed', command = tr, state = st} end
      | SOME st' =>
          let
            val _ = proof_status tr st';
          in {failed = false, malformed = malformed', command = tr, state = st'} end)
    end;

in

fun eval init span eval0 =
  let
    val exec_id = Document_ID.make ();
    fun process () =
      let
        val tr =
          Position.setmp_thread_data (Position.id_only (Document_ID.print exec_id))
            (fn () => read init span |> Toplevel.put_id exec_id) ();
      in eval_state span tr (eval_result eval0) end;
  in {exec_id = exec_id, eval_process = memo process} end;

end;


(* print *)

type print_fn = Toplevel.transition -> Toplevel.state -> unit;

local

type print_function = string * (int * (string -> print_fn option));
val print_functions = Synchronized.var "Command.print_functions" ([]: print_function list);

fun output_error tr exn =
  List.app (Future.error_msg (Toplevel.pos_of tr)) (ML_Compiler.exn_messages_ids exn);

fun print_error tr f x =
  (Toplevel.setmp_thread_position tr o Runtime.controlled_execution) f x
    handle exn => output_error tr exn;

in

fun print command_name eval =
  rev (Synchronized.value print_functions) |> map_filter (fn (name, (pri, get_print_fn)) =>
    (case Exn.capture (Runtime.controlled_execution get_print_fn) command_name of
      Exn.Res NONE => NONE
    | Exn.Res (SOME print_fn) =>
        let
          val exec_id = Document_ID.make ();
          fun process () =
            let
              val {failed, command, state = st', ...} = eval_result eval;
              val tr = Toplevel.put_id exec_id command;
            in if failed then () else print_error tr (fn () => print_fn tr st') () end;
        in SOME {name = name, pri = pri, exec_id = exec_id, print_process = memo process} end
    | Exn.Exn exn =>
        let
          val exec_id = Document_ID.make ();
          fun process () =
            let
              val {command, ...} = eval_result eval;
              val tr = Toplevel.put_id exec_id command;
            in output_error tr exn end;
        in SOME {name = name, pri = pri, exec_id = exec_id, print_process = memo process} end));

fun print_function {name, pri} f =
  Synchronized.change print_functions (fn funs =>
   (if not (AList.defined (op =) funs name) then ()
    else warning ("Redefining command print function: " ^ quote name);
    AList.update (op =) (name, (pri, f)) funs));

end;

val _ =
  print_function {name = "print_state", pri = 0} (fn command_name => SOME (fn tr => fn st' =>
    let
      val is_init = Keyword.is_theory_begin command_name;
      val is_proof = Keyword.is_proof command_name;
      val do_print =
        not is_init andalso
          (Toplevel.print_of tr orelse (is_proof andalso Toplevel.is_proof st'));
    in if do_print then Toplevel.print_state false st' else () end));



(** managed evaluation **)

(* execution *)

type exec = eval * print list;
val no_exec: exec = (no_eval, []);

fun exec_ids (({exec_id, ...}, prints): exec) = exec_id :: map #exec_id prints;

fun exec_run (({eval_process, ...}, prints): exec) =
 (memo_eval eval_process;
  prints |> List.app (fn {name, pri, print_process, ...} =>
    memo_fork {name = name, group = NONE, deps = [], pri = pri, interrupts = true} print_process));


(* stable situations after cancellation *)

fun stable_goals exec_id =
  not (Par_Exn.is_interrupted (Future.join_results (Goal.peek_futures exec_id)));

fun stable_eval ({exec_id, eval_process}: eval) =
  stable_goals exec_id andalso memo_stable eval_process;

fun stable_print ({exec_id, print_process, ...}: print) =
  stable_goals exec_id andalso memo_stable print_process;

end;

