(*  Title:      Pure/Tools/build.ML
    Author:     Makarius

Build Isabelle sessions.
*)

signature BUILD =
sig
  val build: string -> unit
end;

structure Build: BUILD =
struct

(* command timings *)

type timings = ((string * Time.time) Inttab.table) Symtab.table;  (*file -> offset -> name * time *)

val empty_timings: timings = Symtab.empty;

fun update_timings props =
  (case Markup.parse_command_timing_properties props of
    SOME ({file, offset, name}, time) =>
      Symtab.map_default (file, Inttab.empty)
        (Inttab.map_default (offset, (name, time)) (fn (_, t) => (name, Time.+ (t, time))))
  | NONE => I);

fun approximative_id name pos =
  (case (Position.file_of pos, Position.offset_of pos) of
    (SOME file, SOME offset) =>
      if name = "" then NONE else SOME {file = file, offset = offset, name = name}
  | _ => NONE);

fun lookup_timings timings tr =
  (case approximative_id (Toplevel.name_of tr) (Toplevel.pos_of tr) of
    SOME {file, offset, name} =>
      (case Symtab.lookup timings file of
        SOME offsets =>
          (case Inttab.lookup offsets offset of
            SOME (name', time) => if name = name' then SOME time else NONE
          | NONE => NONE)
      | NONE => NONE)
  | NONE => NONE);


(* protocol messages *)

fun inline_message a args =
  writeln ("\f" ^ a ^ " = " ^ YXML.string_of_body (XML.Encode.properties args));

fun protocol_message props output =
  (case props of
    function :: args =>
      if function = Markup.ML_statistics orelse function = Markup.task_statistics then
        inline_message (#2 function) args
      else if function = Markup.command_timing then
        let
          val name = the_default "" (Properties.get args Markup.nameN);
          val pos = Position.of_properties args;
          val {elapsed, ...} = Markup.parse_timing_properties args;
        in
          (case approximative_id name pos of
            SOME id => inline_message (#2 function) (Markup.command_timing_properties id elapsed)
          | NONE => ())
        end
      else
        (case Markup.dest_loading_theory props of
          SOME name => writeln ("\floading_theory = " ^ name)
        | NONE => raise Output.Protocol_Message props)
  | [] => raise Output.Protocol_Message props);


(* build *)

local

fun no_document options =
  (case Options.string options "document" of "" => true | "false" => true | _ => false);

fun use_theories last_timing options =
  Thy_Info.use_theories {last_timing = last_timing, master_dir = Path.current}
    |> Unsynchronized.setmp Proofterm.proofs (Options.int options "proofs")
    |> Unsynchronized.setmp print_mode
        (space_explode "," (Options.string options "print_mode") @ print_mode_value ())
    |> Unsynchronized.setmp Goal.parallel_proofs (Options.int options "parallel_proofs")
    |> Unsynchronized.setmp Goal.parallel_subproofs_saturation
        (Options.int options "parallel_subproofs_saturation")
    |> Unsynchronized.setmp Goal.parallel_subproofs_threshold
        (Options.real options "parallel_subproofs_threshold")
    |> Unsynchronized.setmp Multithreading.trace (Options.int options "threads_trace")
    |> Unsynchronized.setmp Multithreading.max_threads (Options.int options "threads")
    |> Unsynchronized.setmp Future.ML_statistics true
    |> no_document options ? Present.no_document
    |> Unsynchronized.setmp quick_and_dirty (Options.bool options "quick_and_dirty")
    |> Unsynchronized.setmp Goal.skip_proofs (Options.bool options "skip_proofs")
    |> Unsynchronized.setmp Printer.show_question_marks_default
        (Options.bool options "show_question_marks")
    |> Unsynchronized.setmp Name_Space.names_long_default (Options.bool options "names_long")
    |> Unsynchronized.setmp Name_Space.names_short_default (Options.bool options "names_short")
    |> Unsynchronized.setmp Name_Space.names_unique_default (Options.bool options "names_unique")
    |> Unsynchronized.setmp Thy_Output.display_default (Options.bool options "thy_output_display")
    |> Unsynchronized.setmp Thy_Output.quotes_default (Options.bool options "thy_output_quotes")
    |> Unsynchronized.setmp Thy_Output.indent_default (Options.int options "thy_output_indent")
    |> Unsynchronized.setmp Thy_Output.source_default (Options.bool options "thy_output_source")
    |> Unsynchronized.setmp Thy_Output.break_default (Options.bool options "thy_output_break")
    |> Unsynchronized.setmp Pretty.margin_default (Options.int options "pretty_margin")
    |> Unsynchronized.setmp Toplevel.timing (Options.bool options "timing");

fun use_theories_condition last_timing (options, thys) =
  let val condition = space_explode "," (Options.string options "condition") in
    (case filter_out (can getenv_strict) condition of
      [] => use_theories last_timing options (map (rpair Position.none) thys)
    | conds =>
        Output.physical_stderr ("Skipping theories " ^ commas_quote thys ^
          " (undefined " ^ commas conds ^ ")\n"))
  end;

in

fun build args_file = Command_Line.tool (fn () =>
    let
      val (command_timings, (do_output, (options, (verbose, (browser_info,
          (parent_name, (chapter, (name, theories)))))))) =
        File.read (Path.explode args_file) |> YXML.parse_body |>
          let open XML.Decode in
            pair (list properties) (pair bool (pair Options.decode (pair bool (pair string
              (pair string (pair string (pair string
                ((list (pair Options.decode (list string)))))))))))
          end;

      val document_variants =
        map Present.read_variant (space_explode ":" (Options.string options "document_variants"));
      val _ =
        (case duplicates (op =) (map fst document_variants) of
          [] => ()
        | dups => error ("Duplicate document variants: " ^ commas_quote dups));

      val _ = writeln ("\fSession.name = " ^ name);
      val _ =
        Session.init do_output
          (Options.bool options "browser_info")
          (Path.explode browser_info)
          (Options.string options "document")
          (Options.bool options "document_graph")
          (Options.string options "document_output")
          document_variants
          parent_name (chapter, name)
          (false, "")
          verbose;

      val last_timing = lookup_timings (fold update_timings command_timings empty_timings);

      val res1 =
        theories |>
          (List.app (use_theories_condition last_timing)
            |> Session.with_timing name verbose
            |> Unsynchronized.setmp Output.Private_Hooks.protocol_message_fn protocol_message
            |> Unsynchronized.setmp Multithreading.max_threads (Options.int options "threads")
            |> Exn.capture);
      val res2 = Exn.capture Session.finish ();
      val _ = Par_Exn.release_all [res1, res2];

      val _ = if do_output then () else exit 0;
    in 0 end);

end;

end;
