(*  Title:      Pure/Concurrent/task_queue.ML
    Author:     Makarius

Ordered queue of grouped tasks.
*)

signature TASK_QUEUE =
sig
  type group
  val new_group: group option -> group
  val group_id: group -> int
  val eq_group: group * group -> bool
  val cancel_group: group -> exn -> unit
  val is_canceled: group -> bool
  val group_status: group -> exn option
  val str_of_group: group -> string
  val str_of_groups: group -> string
  type task
  val dummy_task: unit -> task
  val group_of_task: task -> group
  val name_of_task: task -> string
  val pri_of_task: task -> int
  val str_of_task: task -> string
  val str_of_task_groups: task -> string
  val timing_of_task: task -> Time.time * Time.time * string list
  val running: task -> (unit -> 'a) -> 'a
  val joining: task -> (unit -> 'a) -> 'a
  val waiting: task -> task list -> (unit -> 'a) -> 'a
  type queue
  val empty: queue
  val known_task: queue -> task -> bool
  val all_passive: queue -> bool
  val status: queue -> {ready: int, pending: int, running: int, passive: int}
  val cancel: queue -> group -> task list * Thread.thread list
  val cancel_all: queue -> group list * Thread.thread list
  val finish: task -> queue -> bool * queue
  val enqueue_passive: group -> (unit -> bool) -> queue -> task * queue
  val enqueue: string -> group -> task list -> int -> (bool -> bool) -> queue -> task * queue
  val extend: task -> (bool -> bool) -> queue -> queue option
  val dequeue_passive: Thread.thread -> task -> queue -> bool * queue
  val dequeue: Thread.thread -> queue -> (task * (bool -> bool) list) option * queue
  val dequeue_deps: Thread.thread -> task list -> queue ->
    (((task * (bool -> bool) list) option * task list) * queue)
end;

structure Task_Queue: TASK_QUEUE =
struct

val new_id = Synchronized.counter ();


(** nested groups of tasks **)

(* groups *)

abstype group = Group of
 {parent: group option,
  id: int,
  status: exn option Synchronized.var}
with

fun make_group (parent, id, status) = Group {parent = parent, id = id, status = status};

fun new_group parent = make_group (parent, new_id (), Synchronized.var "group_status" NONE);

fun group_id (Group {id, ...}) = id;
fun eq_group (group1, group2) = group_id group1 = group_id group2;

fun fold_groups f (g as Group {parent = NONE, ...}) a = f g a
  | fold_groups f (g as Group {parent = SOME group, ...}) a = fold_groups f group (f g a);


(* group status *)

fun cancel_group (Group {status, ...}) exn =
  Synchronized.change status
    (fn exns => SOME (Par_Exn.make (exn :: the_list exns)));

fun is_canceled (Group {parent, status, ...}) =
  is_some (Synchronized.value status) orelse
    (case parent of NONE => false | SOME group => is_canceled group);

fun group_exns (Group {parent, status, ...}) =
  the_list (Synchronized.value status) @
    (case parent of NONE => [] | SOME group => group_exns group);

fun group_status group =
  (case group_exns group of
    [] => NONE
  | exns => SOME (Par_Exn.make exns));

fun str_of_group group =
  (is_canceled group ? enclose "(" ")") (string_of_int (group_id group));

fun str_of_groups group =
  space_implode "/" (map str_of_group (rev (fold_groups cons group [])));

end;


(* tasks *)

type timing = Time.time * Time.time * string list;  (*run, wait, wait dependencies*)

fun new_timing () =
  Synchronized.var "timing" ((Time.zeroTime, Time.zeroTime, []): timing);

abstype task = Task of
 {group: group,
  name: string,
  id: int,
  pri: int option,
  timing: timing Synchronized.var}
with

fun dummy_task () =
  Task {group = new_group NONE, name = "", id = 0, pri = NONE, timing = new_timing ()};

fun new_task group name pri =
  Task {group = group, name = name, id = new_id (), pri = pri, timing = new_timing ()};

fun group_of_task (Task {group, ...}) = group;
fun name_of_task (Task {name, ...}) = name;
fun pri_of_task (Task {pri, ...}) = the_default 0 pri;

fun str_of_task (Task {name, id, ...}) =
  if name = "" then string_of_int id else string_of_int id ^ " (" ^ name ^ ")";

fun str_of_task_groups task = str_of_task task ^ " in " ^ str_of_groups (group_of_task task);

fun timing_of_task (Task {timing, ...}) = Synchronized.value timing;

fun update_timing update (Task {timing, ...}) e =
  uninterruptible (fn restore_attributes => fn () =>
    let
      val start = Time.now ();
      val result = Exn.capture (restore_attributes e) ();
      val t = Time.- (Time.now (), start);
      val _ = Synchronized.change timing (update t);
    in Exn.release result end) ();

fun task_ord (Task {id = id1, pri = pri1, ...}, Task {id = id2, pri = pri2, ...}) =
  prod_ord (rev_order o option_ord int_ord) int_ord ((pri1, id1), (pri2, id2));

end;

structure Tasks = Table(type key = task val ord = task_ord);
structure Task_Graph = Graph(type key = task val ord = task_ord);


(* timing *)

fun running task =
  update_timing (fn t => fn (a, b, ds) => (Time.+ (a, t), b, ds)) task;

fun joining task =
  update_timing (fn t => fn (a, b, ds) => (Time.- (a, t), b, ds)) task;

fun waiting task deps =
  update_timing (fn t => fn (a, b, ds) =>
    (Time.- (a, t), Time.+ (b, t),
      if ! Multithreading.trace > 0
      then fold (insert (op =) o name_of_task) deps ds else ds)) task;



(** queue of jobs and groups **)

(* known group members *)

type groups = unit Tasks.table Inttab.table;

fun get_tasks (groups: groups) gid =
  the_default Tasks.empty (Inttab.lookup groups gid);

fun add_task (gid, task) groups =
  Inttab.update (gid, Tasks.update (task, ()) (get_tasks groups gid)) groups;

fun del_task (gid, task) groups =
  let val tasks = Tasks.delete_safe task (get_tasks groups gid) in
    if Tasks.is_empty tasks then Inttab.delete_safe gid groups
    else Inttab.update (gid, tasks) groups
  end;


(* job dependency graph *)

datatype job =
  Job of (bool -> bool) list |
  Running of Thread.thread |
  Passive of unit -> bool;

type jobs = job Task_Graph.T;

fun get_job (jobs: jobs) task = Task_Graph.get_node jobs task;
fun set_job task job (jobs: jobs) = Task_Graph.map_node task (K job) jobs;

fun add_job task dep (jobs: jobs) =
  Task_Graph.add_edge (dep, task) jobs handle Task_Graph.UNDEF _ => jobs;


(* queue *)

datatype queue = Queue of {groups: groups, jobs: jobs};

fun make_queue groups jobs = Queue {groups = groups, jobs = jobs};
val empty = make_queue Inttab.empty Task_Graph.empty;

fun known_task (Queue {jobs, ...}) task = can (Task_Graph.get_entry jobs) task;


(* job status *)

fun ready_job task (Job list, (deps, _)) =
      if Task_Graph.Keys.is_empty deps then SOME (task, rev list) else NONE
  | ready_job task (Passive abort, (deps, _)) =
      if Task_Graph.Keys.is_empty deps andalso is_canceled (group_of_task task)
      then SOME (task, [fn _ => abort ()])
      else NONE
  | ready_job _ _ = NONE;

fun active_job (_, (Job _, _)) = SOME ()
  | active_job (_, (Running _, _)) = SOME ()
  | active_job (task, (Passive _, _)) =
      if is_canceled (group_of_task task) then SOME () else NONE;

fun all_passive (Queue {jobs, ...}) = is_none (Task_Graph.get_first active_job jobs);


(* queue status *)

fun status (Queue {jobs, ...}) =
  let
    val (x, y, z, w) =
      Task_Graph.fold (fn (_, (job, (deps, _))) => fn (x, y, z, w) =>
          (case job of
            Job _ => if Task_Graph.Keys.is_empty deps then (x + 1, y, z, w) else (x, y + 1, z, w)
          | Running _ => (x, y, z + 1, w)
          | Passive _ => (x, y, z, w + 1)))
        jobs (0, 0, 0, 0);
  in {ready = x, pending = y, running = z, passive = w} end;



(** task queue operations **)

(* cancel -- peers and sub-groups *)

fun cancel (Queue {groups, jobs}) group =
  let
    val _ = cancel_group group Exn.Interrupt;
    val running =
      Tasks.fold (fn (task, _) => fn (tasks, threads) =>
          (case get_job jobs task of
            Running thread => (task :: tasks, insert Thread.equal thread threads)
          | Passive _ => (task :: tasks, threads)
          | _ => (tasks, threads)))
        (get_tasks groups (group_id group)) ([], []);
  in running end;

fun cancel_all (Queue {jobs, ...}) =
  let
    fun cancel_job (task, (job, _)) (groups, running) =
      let
        val group = group_of_task task;
        val _ = cancel_group group Exn.Interrupt;
      in
        (case job of
          Running t => (insert eq_group group groups, insert Thread.equal t running)
        | _ => (groups, running))
      end;
    val running = Task_Graph.fold cancel_job jobs ([], []);
  in running end;


(* finish *)

fun finish task (Queue {groups, jobs}) =
  let
    val group = group_of_task task;
    val groups' = fold_groups (fn g => del_task (group_id g, task)) group groups;
    val jobs' = Task_Graph.del_node task jobs;
    val maximal = Task_Graph.is_maximal jobs task;
  in (maximal, make_queue groups' jobs') end;


(* enqueue *)

fun enqueue_passive group abort (Queue {groups, jobs}) =
  let
    val task = new_task group "passive" NONE;
    val groups' = fold_groups (fn g => add_task (group_id g, task)) group groups;
    val jobs' = jobs |> Task_Graph.new_node (task, Passive abort);
  in (task, make_queue groups' jobs') end;

fun enqueue name group deps pri job (Queue {groups, jobs}) =
  let
    val task = new_task group name (SOME pri);
    val groups' = fold_groups (fn g => add_task (group_id g, task)) group groups;
    val jobs' = jobs
      |> Task_Graph.new_node (task, Job [job])
      |> fold (add_job task) deps;
  in (task, make_queue groups' jobs') end;

fun extend task job (Queue {groups, jobs}) =
  (case try (get_job jobs) task of
    SOME (Job list) => SOME (make_queue groups (set_job task (Job (job :: list)) jobs))
  | _ => NONE);


(* dequeue *)

fun dequeue_passive thread task (queue as Queue {groups, jobs}) =
  (case try (get_job jobs) task of
    SOME (Passive _) =>
      let val jobs' = set_job task (Running thread) jobs
      in (true, make_queue groups jobs') end
  | _ => (false, queue));

fun dequeue thread (queue as Queue {groups, jobs}) =
  (case Task_Graph.get_first (uncurry ready_job) jobs of
    SOME (result as (task, _)) =>
      let val jobs' = set_job task (Running thread) jobs
      in (SOME result, make_queue groups jobs') end
  | NONE => (NONE, queue));


(* dequeue wrt. dynamic dependencies *)

fun dequeue_deps thread deps (queue as Queue {groups, jobs}) =
  let
    fun ready [] rest = (NONE, rev rest)
      | ready (task :: tasks) rest =
          (case try (Task_Graph.get_entry jobs) task of
            NONE => ready tasks rest
          | SOME (_, entry) =>
              (case ready_job task entry of
                NONE => ready tasks (task :: rest)
              | some => (some, List.revAppend (rest, tasks))));

    fun ready_dep _ [] = NONE
      | ready_dep seen (task :: tasks) =
          if Tasks.defined seen task then ready_dep seen tasks
          else
            let val entry as (_, (ds, _)) = #2 (Task_Graph.get_entry jobs task) in
              (case ready_job task entry of
                NONE => ready_dep (Tasks.update (task, ()) seen) (Task_Graph.Keys.dest ds @ tasks)
              | some => some)
            end;

    fun result (res as (task, _)) deps' =
      let val jobs' = set_job task (Running thread) jobs
      in ((SOME res, deps'), make_queue groups jobs') end;
  in
    (case ready deps [] of
      (SOME res, deps') => result res deps'
    | (NONE, deps') =>
        (case ready_dep Tasks.empty deps' of
          SOME res => result res deps'
        | NONE => ((NONE, deps'), queue)))
  end;

end;
