(*  Title:      HOL/SET_Protocol/Purchase.thy
    Author:     Giampaolo Bella
    Author:     Fabio Massacci
    Author:     Lawrence C Paulson
*)

section\<open>Purchase Phase of SET\<close>

theory Purchase
imports Public_SET
begin

text\<open>
Note: nonces seem to consist of 20 bytes.  That includes both freshness
challenges (Chall-EE, etc.) and important secrets (CardSecret, PANsecret)

This version omits \<open>LID_C\<close> but retains \<open>LID_M\<close>. At first glance
(Programmer's Guide page 267) it seems that both numbers are just introduced
for the respective convenience of the Cardholder's and Merchant's
system. However, omitting both of them would create a problem of
identification: how can the Merchant's system know what transaction is it
supposed to process?

Further reading (Programmer's guide page 309) suggest that there is an outside
bootstrapping message (SET initiation message) which is used by the Merchant
and the Cardholder to agree on the actual transaction. This bootstrapping
message is described in the SET External Interface Guide and ought to generate
\<open>LID_M\<close>. According SET Extern Interface Guide, this number might be a
cookie, an invoice number etc. The Programmer's Guide on page 310, states that
in absence of \<open>LID_M\<close> the protocol must somehow ("outside SET") identify
the transaction from OrderDesc, which is assumed to be a searchable text only
field. Thus, it is assumed that the Merchant or the Cardholder somehow agreed
out-of-bad on the value of \<open>LID_M\<close> (for instance a cookie in a web
transaction etc.). This out-of-band agreement is expressed with a preliminary
start action in which the merchant and the Cardholder agree on the appropriate
values. Agreed values are stored with a suitable notes action.

"XID is a transaction ID that is usually generated by the Merchant system,
unless there is no PInitRes, in which case it is generated by the Cardholder
system. It is a randomly generated 20 byte variable that is globally unique
(statistically). Merchant and Cardholder systems shall use appropriate random
number generators to ensure the global uniqueness of XID."
--Programmer's Guide, page 267.

PI (Payment Instruction) is the most central and sensitive data structure in
SET. It is used to pass the data required to authorize a payment card payment
from the Cardholder to the Payment Gateway, which will use the data to
initiate a payment card transaction through the traditional payment card
financial network. The data is encrypted by the Cardholder and sent via the
Merchant, such that the data is hidden from the Merchant unless the Acquirer
passes the data back to the Merchant.
--Programmer's Guide, page 271.\<close>

consts

    CardSecret :: "nat => nat"
     \<comment>\<open>Maps Cardholders to CardSecrets.
         A CardSecret of 0 means no cerificate, must use unsigned format.\<close>

    PANSecret :: "nat => nat"
     \<comment>\<open>Maps Cardholders to PANSecrets.\<close>

inductive_set
  set_pur :: "event list set"
where

  Nil:   \<comment>\<open>Initial trace is empty\<close>
         "[] \<in> set_pur"

| Fake:  \<comment>\<open>The spy MAY say anything he CAN say.\<close>
         "[| evsf \<in> set_pur;  X \<in> synth(analz(knows Spy evsf)) |]
          ==> Says Spy B X  # evsf \<in> set_pur"


| Reception: \<comment>\<open>If A sends a message X to B, then B might receive it\<close>
             "[| evsr \<in> set_pur;  Says A B X \<in> set evsr |]
              ==> Gets B X  # evsr \<in> set_pur"

| Start: 
      \<comment>\<open>Added start event which is out-of-band for SET: the Cardholder and
          the merchant agree on the amounts and uses \<open>LID_M\<close> as an
          identifier.
          This is suggested by the External Interface Guide. The Programmer's
          Guide, in absence of \<open>LID_M\<close>, states that the merchant uniquely
          identifies the order out of some data contained in OrderDesc.\<close>
   "[|evsStart \<in> set_pur;
      Number LID_M \<notin> used evsStart;
      C = Cardholder k; M = Merchant i; P = PG j;
      Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
      LID_M \<notin> range CardSecret;
      LID_M \<notin> range PANSecret |]
     ==> Notes C \<lbrace>Number LID_M, Transaction\<rbrace>
       # Notes M \<lbrace>Number LID_M, Agent P, Transaction\<rbrace>
       # evsStart \<in> set_pur"

| PInitReq:
     \<comment>\<open>Purchase initialization, page 72 of Formal Protocol Desc.\<close>
   "[|evsPIReq \<in> set_pur;
      Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
      Nonce Chall_C \<notin> used evsPIReq;
      Chall_C \<notin> range CardSecret; Chall_C \<notin> range PANSecret;
      Notes C \<lbrace>Number LID_M, Transaction \<rbrace> \<in> set evsPIReq |]
    ==> Says C M \<lbrace>Number LID_M, Nonce Chall_C\<rbrace> # evsPIReq \<in> set_pur"

| PInitRes:
     \<comment>\<open>Merchant replies with his own label XID and the encryption
         key certificate of his chosen Payment Gateway. Page 74 of Formal
         Protocol Desc. We use \<open>LID_M\<close> to identify Cardholder\<close>
   "[|evsPIRes \<in> set_pur;
      Gets M \<lbrace>Number LID_M, Nonce Chall_C\<rbrace> \<in> set evsPIRes;
      Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
      Notes M \<lbrace>Number LID_M, Agent P, Transaction\<rbrace> \<in> set evsPIRes;
      Nonce Chall_M \<notin> used evsPIRes;
      Chall_M \<notin> range CardSecret; Chall_M \<notin> range PANSecret;
      Number XID \<notin> used evsPIRes;
      XID \<notin> range CardSecret; XID \<notin> range PANSecret|]
    ==> Says M C (sign (priSK M)
                       \<lbrace>Number LID_M, Number XID,
                         Nonce Chall_C, Nonce Chall_M,
                         cert P (pubEK P) onlyEnc (priSK RCA)\<rbrace>)
          # evsPIRes \<in> set_pur"

| PReqUns:
      \<comment>\<open>UNSIGNED Purchase request (CardSecret = 0).
        Page 79 of Formal Protocol Desc.
        Merchant never sees the amount in clear. This holds of the real
        protocol, where XID identifies the transaction. We omit
        \<open>Hash\<lbrace>Number XID, Nonce (CardSecret k)\<rbrace>\<close> from PIHead because
        the CardSecret is 0 and because AuthReq treated the unsigned case
        very differently from the signed one anyway.\<close>
   "!!Chall_C Chall_M OrderDesc P PurchAmt XID evsPReqU.
    [|evsPReqU \<in> set_pur;
      C = Cardholder k; CardSecret k = 0;
      Key KC1 \<notin> used evsPReqU;  KC1 \<in> symKeys;
      Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
      HOD = Hash\<lbrace>Number OrderDesc, Number PurchAmt\<rbrace>;
      OIData = \<lbrace>Number LID_M, Number XID, Nonce Chall_C, HOD,Nonce Chall_M\<rbrace>;
      PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M\<rbrace>;
      Gets C (sign (priSK M)
                   \<lbrace>Number LID_M, Number XID,
                     Nonce Chall_C, Nonce Chall_M,
                     cert P EKj onlyEnc (priSK RCA)\<rbrace>)
        \<in> set evsPReqU;
      Says C M \<lbrace>Number LID_M, Nonce Chall_C\<rbrace> \<in> set evsPReqU;
      Notes C \<lbrace>Number LID_M, Transaction\<rbrace> \<in> set evsPReqU |]
    ==> Says C M
             \<lbrace>EXHcrypt KC1 EKj \<lbrace>PIHead, Hash OIData\<rbrace> (Pan (pan C)),
               OIData, Hash\<lbrace>PIHead, Pan (pan C)\<rbrace> \<rbrace>
          # Notes C \<lbrace>Key KC1, Agent M\<rbrace>
          # evsPReqU \<in> set_pur"

| PReqS:
      \<comment>\<open>SIGNED Purchase request.  Page 77 of Formal Protocol Desc.
          We could specify the equation
          @{term "PIReqSigned = \<lbrace> PIDualSigned, OIDualSigned \<rbrace>"}, since the
          Formal Desc. gives PIHead the same format in the unsigned case.
          However, there's little point, as P treats the signed and 
          unsigned cases differently.\<close>
   "!!C Chall_C Chall_M EKj HOD KC2 LID_M M OIData
      OIDualSigned OrderDesc P PANData PIData PIDualSigned
      PIHead PurchAmt Transaction XID evsPReqS k.
    [|evsPReqS \<in> set_pur;
      C = Cardholder k;
      CardSecret k \<noteq> 0;  Key KC2 \<notin> used evsPReqS;  KC2 \<in> symKeys;
      Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
      HOD = Hash\<lbrace>Number OrderDesc, Number PurchAmt\<rbrace>;
      OIData = \<lbrace>Number LID_M, Number XID, Nonce Chall_C, HOD, Nonce Chall_M\<rbrace>;
      PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M,
                  Hash\<lbrace>Number XID, Nonce (CardSecret k)\<rbrace>\<rbrace>;
      PANData = \<lbrace>Pan (pan C), Nonce (PANSecret k)\<rbrace>;
      PIData = \<lbrace>PIHead, PANData\<rbrace>;
      PIDualSigned = \<lbrace>sign (priSK C) \<lbrace>Hash PIData, Hash OIData\<rbrace>,
                       EXcrypt KC2 EKj \<lbrace>PIHead, Hash OIData\<rbrace> PANData\<rbrace>;
      OIDualSigned = \<lbrace>OIData, Hash PIData\<rbrace>;
      Gets C (sign (priSK M)
                   \<lbrace>Number LID_M, Number XID,
                     Nonce Chall_C, Nonce Chall_M,
                     cert P EKj onlyEnc (priSK RCA)\<rbrace>)
        \<in> set evsPReqS;
      Says C M \<lbrace>Number LID_M, Nonce Chall_C\<rbrace> \<in> set evsPReqS;
      Notes C \<lbrace>Number LID_M, Transaction\<rbrace> \<in> set evsPReqS |]
    ==> Says C M \<lbrace>PIDualSigned, OIDualSigned\<rbrace>
          # Notes C \<lbrace>Key KC2, Agent M\<rbrace>
          # evsPReqS \<in> set_pur"

  \<comment>\<open>Authorization Request.  Page 92 of Formal Protocol Desc.
    Sent in response to Purchase Request.\<close>
| AuthReq:
   "[| evsAReq \<in> set_pur;
       Key KM \<notin> used evsAReq;  KM \<in> symKeys;
       Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
       HOD = Hash\<lbrace>Number OrderDesc, Number PurchAmt\<rbrace>;
       OIData = \<lbrace>Number LID_M, Number XID, Nonce Chall_C, HOD,
                  Nonce Chall_M\<rbrace>;
       CardSecret k \<noteq> 0 -->
         P_I = \<lbrace>sign (priSK C) \<lbrace>HPIData, Hash OIData\<rbrace>, encPANData\<rbrace>;
       Gets M \<lbrace>P_I, OIData, HPIData\<rbrace> \<in> set evsAReq;
       Says M C (sign (priSK M) \<lbrace>Number LID_M, Number XID,
                                  Nonce Chall_C, Nonce Chall_M,
                                  cert P EKj onlyEnc (priSK RCA)\<rbrace>)
         \<in> set evsAReq;
        Notes M \<lbrace>Number LID_M, Agent P, Transaction\<rbrace>
           \<in> set evsAReq |]
    ==> Says M P
             (EncB (priSK M) KM (pubEK P)
               \<lbrace>Number LID_M, Number XID, Hash OIData, HOD\<rbrace>   P_I)
          # evsAReq \<in> set_pur"

  \<comment>\<open>Authorization Response has two forms: for UNSIGNED and SIGNED PIs.
    Page 99 of Formal Protocol Desc.
    PI is a keyword (product!), so we call it \<open>P_I\<close>. The hashes HOD and
    HOIData occur independently in \<open>P_I\<close> and in M's message.
    The authCode in AuthRes represents the baggage of EncB, which in the
    full protocol is [CapToken], [AcqCardMsg], [AuthToken]:
    optional items for split shipments, recurring payments, etc.\<close>

| AuthResUns:
    \<comment>\<open>Authorization Response, UNSIGNED\<close>
   "[| evsAResU \<in> set_pur;
       C = Cardholder k; M = Merchant i;
       Key KP \<notin> used evsAResU;  KP \<in> symKeys;
       CardSecret k = 0;  KC1 \<in> symKeys;  KM \<in> symKeys;
       PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M\<rbrace>;
       P_I = EXHcrypt KC1 EKj \<lbrace>PIHead, HOIData\<rbrace> (Pan (pan C));
       Gets P (EncB (priSK M) KM (pubEK P)
               \<lbrace>Number LID_M, Number XID, HOIData, HOD\<rbrace> P_I)
           \<in> set evsAResU |]
   ==> Says P M
            (EncB (priSK P) KP (pubEK M)
              \<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>
              authCode)
       # evsAResU \<in> set_pur"

| AuthResS:
    \<comment>\<open>Authorization Response, SIGNED\<close>
   "[| evsAResS \<in> set_pur;
       C = Cardholder k;
       Key KP \<notin> used evsAResS;  KP \<in> symKeys;
       CardSecret k \<noteq> 0;  KC2 \<in> symKeys;  KM \<in> symKeys;
       P_I = \<lbrace>sign (priSK C) \<lbrace>Hash PIData, HOIData\<rbrace>,
               EXcrypt KC2 (pubEK P) \<lbrace>PIHead, HOIData\<rbrace> PANData\<rbrace>;
       PANData = \<lbrace>Pan (pan C), Nonce (PANSecret k)\<rbrace>;
       PIData = \<lbrace>PIHead, PANData\<rbrace>;
       PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M,
                  Hash\<lbrace>Number XID, Nonce (CardSecret k)\<rbrace>\<rbrace>;
       Gets P (EncB (priSK M) KM (pubEK P)
                \<lbrace>Number LID_M, Number XID, HOIData, HOD\<rbrace>
               P_I)
           \<in> set evsAResS |]
   ==> Says P M
            (EncB (priSK P) KP (pubEK M)
              \<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>
              authCode)
       # evsAResS \<in> set_pur"

| PRes:
    \<comment>\<open>Purchase response.\<close>
   "[| evsPRes \<in> set_pur;  KP \<in> symKeys;  M = Merchant i;
       Transaction = \<lbrace>Agent M, Agent C, Number OrderDesc, Number PurchAmt\<rbrace>;
       Gets M (EncB (priSK P) KP (pubEK M)
              \<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>
              authCode)
          \<in> set evsPRes;
       Gets M \<lbrace>Number LID_M, Nonce Chall_C\<rbrace> \<in> set evsPRes;
       Says M P
            (EncB (priSK M) KM (pubEK P)
              \<lbrace>Number LID_M, Number XID, Hash OIData, HOD\<rbrace> P_I)
         \<in> set evsPRes;
       Notes M \<lbrace>Number LID_M, Agent P, Transaction\<rbrace>
          \<in> set evsPRes
      |]
   ==> Says M C
         (sign (priSK M) \<lbrace>Number LID_M, Number XID, Nonce Chall_C,
                           Hash (Number PurchAmt)\<rbrace>)
         # evsPRes \<in> set_pur"


specification (CardSecret PANSecret)
  inj_CardSecret:  "inj CardSecret"
  inj_PANSecret:   "inj PANSecret"
  CardSecret_neq_PANSecret: "CardSecret k \<noteq> PANSecret k'"
    \<comment>\<open>No CardSecret equals any PANSecret\<close>
  apply (rule_tac x="curry prod_encode 0" in exI)
  apply (rule_tac x="curry prod_encode 1" in exI)
  apply (simp add: prod_encode_eq inj_on_def)
  done

declare Says_imp_knows_Spy [THEN parts.Inj, dest]
declare parts.Body [dest]
declare analz_into_parts [dest]
declare Fake_parts_insert_in_Un [dest]

declare CardSecret_neq_PANSecret [iff] 
        CardSecret_neq_PANSecret [THEN not_sym, iff]
declare inj_CardSecret [THEN inj_eq, iff] 
        inj_PANSecret [THEN inj_eq, iff]


subsection\<open>Possibility Properties\<close>

lemma Says_to_Gets:
     "Says A B X # evs \<in> set_pur ==> Gets B X # Says A B X # evs \<in> set_pur"
by (rule set_pur.Reception, auto)

text\<open>Possibility for UNSIGNED purchases. Note that we need to ensure
that XID differs from OrderDesc and PurchAmt, since it is supposed to be
a unique number!\<close>
lemma possibility_Uns:
    "[| CardSecret k = 0;
        C = Cardholder k;  M = Merchant i;
        Key KC \<notin> used []; Key KM \<notin> used []; Key KP \<notin> used []; 
        KC \<in> symKeys; KM \<in> symKeys; KP \<in> symKeys; 
        KC < KM; KM < KP;
        Nonce Chall_C \<notin> used []; Chall_C \<notin> range CardSecret \<union> range PANSecret;
        Nonce Chall_M \<notin> used []; Chall_M \<notin> range CardSecret \<union> range PANSecret;
        Chall_C < Chall_M; 
        Number LID_M \<notin> used []; LID_M \<notin> range CardSecret \<union> range PANSecret;
        Number XID \<notin> used []; XID \<notin> range CardSecret \<union> range PANSecret;
        LID_M < XID; XID < OrderDesc; OrderDesc < PurchAmt |] 
   ==> \<exists>evs \<in> set_pur.
          Says M C
               (sign (priSK M)
                    \<lbrace>Number LID_M, Number XID, Nonce Chall_C, 
                      Hash (Number PurchAmt)\<rbrace>)
                  \<in> set evs" 
apply (intro exI bexI)
apply (rule_tac [2]
        set_pur.Nil
         [THEN set_pur.Start [of _ LID_M C k M i _ _ _ OrderDesc PurchAmt], 
          THEN set_pur.PInitReq [of concl: C M LID_M Chall_C],
          THEN Says_to_Gets, 
          THEN set_pur.PInitRes [of concl: M C LID_M XID Chall_C Chall_M], 
          THEN Says_to_Gets,
          THEN set_pur.PReqUns [of concl: C M KC],
          THEN Says_to_Gets, 
          THEN set_pur.AuthReq [of concl: M "PG j" KM LID_M XID], 
          THEN Says_to_Gets, 
          THEN set_pur.AuthResUns [of concl: "PG j" M KP LID_M XID],
          THEN Says_to_Gets, 
          THEN set_pur.PRes]) 
apply basic_possibility
apply (simp_all add: used_Cons symKeys_neq_imp_neq) 
done

lemma possibility_S:
    "[| CardSecret k \<noteq> 0;
        C = Cardholder k;  M = Merchant i;
        Key KC \<notin> used []; Key KM \<notin> used []; Key KP \<notin> used []; 
        KC \<in> symKeys; KM \<in> symKeys; KP \<in> symKeys; 
        KC < KM; KM < KP;
        Nonce Chall_C \<notin> used []; Chall_C \<notin> range CardSecret \<union> range PANSecret;
        Nonce Chall_M \<notin> used []; Chall_M \<notin> range CardSecret \<union> range PANSecret;
        Chall_C < Chall_M; 
        Number LID_M \<notin> used []; LID_M \<notin> range CardSecret \<union> range PANSecret;
        Number XID \<notin> used []; XID \<notin> range CardSecret \<union> range PANSecret;
        LID_M < XID; XID < OrderDesc; OrderDesc < PurchAmt |] 
   ==>  \<exists>evs \<in> set_pur.
            Says M C
                 (sign (priSK M) \<lbrace>Number LID_M, Number XID, Nonce Chall_C, 
                                   Hash (Number PurchAmt)\<rbrace>)
               \<in> set evs"
apply (intro exI bexI)
apply (rule_tac [2]
        set_pur.Nil
         [THEN set_pur.Start [of _ LID_M C k M i _ _ _ OrderDesc PurchAmt], 
          THEN set_pur.PInitReq [of concl: C M LID_M Chall_C],
          THEN Says_to_Gets, 
          THEN set_pur.PInitRes [of concl: M C LID_M XID Chall_C Chall_M], 
          THEN Says_to_Gets,
          THEN set_pur.PReqS [of concl: C M _ _ KC],
          THEN Says_to_Gets, 
          THEN set_pur.AuthReq [of concl: M "PG j" KM LID_M XID], 
          THEN Says_to_Gets, 
          THEN set_pur.AuthResS [of concl: "PG j" M KP LID_M XID],
          THEN Says_to_Gets, 
          THEN set_pur.PRes]) 
apply basic_possibility
apply (auto simp add: used_Cons symKeys_neq_imp_neq) 
done

text\<open>General facts about message reception\<close>
lemma Gets_imp_Says:
     "[| Gets B X \<in> set evs; evs \<in> set_pur |]
   ==> \<exists>A. Says A B X \<in> set evs"
apply (erule rev_mp)
apply (erule set_pur.induct, auto)
done

lemma Gets_imp_knows_Spy:
     "[| Gets B X \<in> set evs; evs \<in> set_pur |]  ==> X \<in> knows Spy evs"
by (blast dest!: Gets_imp_Says Says_imp_knows_Spy)

declare Gets_imp_knows_Spy [THEN parts.Inj, dest]

text\<open>Forwarding lemmas, to aid simplification\<close>

lemma AuthReq_msg_in_parts_spies:
     "[|Gets M \<lbrace>P_I, OIData, HPIData\<rbrace> \<in> set evs;
        evs \<in> set_pur|] ==> P_I \<in> parts (knows Spy evs)"
by auto

lemma AuthReq_msg_in_analz_spies:
     "[|Gets M \<lbrace>P_I, OIData, HPIData\<rbrace> \<in> set evs;
        evs \<in> set_pur|] ==> P_I \<in> analz (knows Spy evs)"
by (blast dest: Gets_imp_knows_Spy [THEN analz.Inj])


subsection\<open>Proofs on Asymmetric Keys\<close>

text\<open>Private Keys are Secret\<close>

text\<open>Spy never sees an agent's private keys! (unless it's bad at start)\<close>
lemma Spy_see_private_Key [simp]:
     "evs \<in> set_pur
      ==> (Key(invKey (publicKey b A)) \<in> parts(knows Spy evs)) = (A \<in> bad)"
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply auto
done
declare Spy_see_private_Key [THEN [2] rev_iffD1, dest!]

lemma Spy_analz_private_Key [simp]:
     "evs \<in> set_pur ==>
     (Key(invKey (publicKey b A)) \<in> analz(knows Spy evs)) = (A \<in> bad)"
by auto
declare Spy_analz_private_Key [THEN [2] rev_iffD1, dest!]

text\<open>rewriting rule for priEK's\<close>
lemma parts_image_priEK:
     "[|Key (priEK C) \<in> parts (Key`KK Un (knows Spy evs));
        evs \<in> set_pur|] ==> priEK C \<in> KK | C \<in> bad"
by auto

text\<open>trivial proof because @{term"priEK C"} never appears even in
  @{term "parts evs"}.\<close>
lemma analz_image_priEK:
     "evs \<in> set_pur ==>
          (Key (priEK C) \<in> analz (Key`KK Un (knows Spy evs))) =
          (priEK C \<in> KK | C \<in> bad)"
by (blast dest!: parts_image_priEK intro: analz_mono [THEN [2] rev_subsetD])


subsection\<open>Public Keys in Certificates are Correct\<close>

lemma Crypt_valid_pubEK [dest!]:
     "[| Crypt (priSK RCA) \<lbrace>Agent C, Key EKi, onlyEnc\<rbrace>
           \<in> parts (knows Spy evs);
         evs \<in> set_pur |] ==> EKi = pubEK C"
by (erule rev_mp, erule set_pur.induct, auto)

lemma Crypt_valid_pubSK [dest!]:
     "[| Crypt (priSK RCA) \<lbrace>Agent C, Key SKi, onlySig\<rbrace>
           \<in> parts (knows Spy evs);
         evs \<in> set_pur |] ==> SKi = pubSK C"
by (erule rev_mp, erule set_pur.induct, auto)

lemma certificate_valid_pubEK:
    "[| cert C EKi onlyEnc (priSK RCA) \<in> parts (knows Spy evs);
        evs \<in> set_pur |]
     ==> EKi = pubEK C"
by (unfold cert_def signCert_def, auto)

lemma certificate_valid_pubSK:
    "[| cert C SKi onlySig (priSK RCA) \<in> parts (knows Spy evs);
        evs \<in> set_pur |] ==> SKi = pubSK C"
by (unfold cert_def signCert_def, auto)

lemma Says_certificate_valid [simp]:
     "[| Says A B (sign SK \<lbrace>lid, xid, cc, cm,
                           cert C EK onlyEnc (priSK RCA)\<rbrace>) \<in> set evs;
         evs \<in> set_pur |]
      ==> EK = pubEK C"
by (unfold sign_def, auto)

lemma Gets_certificate_valid [simp]:
     "[| Gets A (sign SK \<lbrace>lid, xid, cc, cm,
                           cert C EK onlyEnc (priSK RCA)\<rbrace>) \<in> set evs;
         evs \<in> set_pur |]
      ==> EK = pubEK C"
by (frule Gets_imp_Says, auto)

method_setup valid_certificate_tac = \<open>
  Args.goal_spec >> (fn quant =>
    fn ctxt => SIMPLE_METHOD'' quant (fn i =>
      EVERY [forward_tac ctxt @{thms Gets_certificate_valid} i,
             assume_tac ctxt i, REPEAT (hyp_subst_tac ctxt i)]))
\<close>


subsection\<open>Proofs on Symmetric Keys\<close>

text\<open>Nobody can have used non-existent keys!\<close>
lemma new_keys_not_used [rule_format,simp]:
     "evs \<in> set_pur
      ==> Key K \<notin> used evs --> K \<in> symKeys -->
          K \<notin> keysFor (parts (knows Spy evs))"
apply (erule set_pur.induct)
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (valid_certificate_tac [7]) \<comment>\<open>PReqUns\<close>
apply auto
apply (force dest!: usedI keysFor_parts_insert) \<comment>\<open>Fake\<close>
done

lemma new_keys_not_analzd:
     "[|Key K \<notin> used evs; K \<in> symKeys; evs \<in> set_pur |]
      ==> K \<notin> keysFor (analz (knows Spy evs))"
by (blast intro: keysFor_mono [THEN [2] rev_subsetD] dest: new_keys_not_used)

lemma Crypt_parts_imp_used:
     "[|Crypt K X \<in> parts (knows Spy evs);
        K \<in> symKeys; evs \<in> set_pur |] ==> Key K \<in> used evs"
apply (rule ccontr)
apply (force dest: new_keys_not_used Crypt_imp_invKey_keysFor)
done

lemma Crypt_analz_imp_used:
     "[|Crypt K X \<in> analz (knows Spy evs);
        K \<in> symKeys; evs \<in> set_pur |] ==> Key K \<in> used evs"
by (blast intro: Crypt_parts_imp_used)

text\<open>New versions: as above, but generalized to have the KK argument\<close>

lemma gen_new_keys_not_used:
     "[|Key K \<notin> used evs; K \<in> symKeys; evs \<in> set_pur |]
      ==> Key K \<notin> used evs --> K \<in> symKeys -->
          K \<notin> keysFor (parts (Key`KK Un knows Spy evs))"
by auto

lemma gen_new_keys_not_analzd:
     "[|Key K \<notin> used evs; K \<in> symKeys; evs \<in> set_pur |]
      ==> K \<notin> keysFor (analz (Key`KK Un knows Spy evs))"
by (blast intro: keysFor_mono [THEN subsetD] dest: gen_new_keys_not_used)

lemma analz_Key_image_insert_eq:
     "[|Key K \<notin> used evs; K \<in> symKeys; evs \<in> set_pur |]
      ==> analz (Key ` (insert K KK) \<union> knows Spy evs) =
          insert (Key K) (analz (Key ` KK \<union> knows Spy evs))"
by (simp add: gen_new_keys_not_analzd)


subsection\<open>Secrecy of Symmetric Keys\<close>

lemma Key_analz_image_Key_lemma:
     "P --> (Key K \<in> analz (Key`KK Un H)) --> (K\<in>KK | Key K \<in> analz H)
      ==>
      P --> (Key K \<in> analz (Key`KK Un H)) = (K\<in>KK | Key K \<in> analz H)"
by (blast intro: analz_mono [THEN [2] rev_subsetD])


lemma symKey_compromise:
     "evs \<in> set_pur \<Longrightarrow>
      (\<forall>SK KK. SK \<in> symKeys \<longrightarrow>
        (\<forall>K \<in> KK. K \<notin> range(\<lambda>C. priEK C)) \<longrightarrow>
               (Key SK \<in> analz (Key`KK \<union> (knows Spy evs))) =
               (SK \<in> KK \<or> Key SK \<in> analz (knows Spy evs)))"
apply (erule set_pur.induct)
apply (rule_tac [!] allI)+
apply (rule_tac [!] impI [THEN Key_analz_image_Key_lemma, THEN impI])+
apply (frule_tac [9] AuthReq_msg_in_analz_spies) \<comment>\<open>AReq\<close>
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (valid_certificate_tac [7]) \<comment>\<open>PReqUns\<close>
apply (simp_all
         del: image_insert image_Un imp_disjL
         add: analz_image_keys_simps disj_simps
              analz_Key_image_insert_eq notin_image_iff
              analz_insert_simps analz_image_priEK)
  \<comment>\<open>8 seconds on a 1.6GHz machine\<close>
apply spy_analz \<comment>\<open>Fake\<close>
apply (blast elim!: ballE)+ \<comment>\<open>PReq: unsigned and signed\<close>
done



subsection\<open>Secrecy of Nonces\<close>

text\<open>As usual: we express the property as a logical equivalence\<close>
lemma Nonce_analz_image_Key_lemma:
     "P --> (Nonce N \<in> analz (Key`KK Un H)) --> (Nonce N \<in> analz H)
      ==> P --> (Nonce N \<in> analz (Key`KK Un H)) = (Nonce N \<in> analz H)"
by (blast intro: analz_mono [THEN [2] rev_subsetD])

text\<open>The \<open>(no_asm)\<close> attribute is essential, since it retains
  the quantifier and allows the simprule's condition to itself be simplified.\<close>
lemma Nonce_compromise [rule_format (no_asm)]:
     "evs \<in> set_pur ==>
      (\<forall>N KK. (\<forall>K \<in> KK. K \<notin> range(\<lambda>C. priEK C))   -->
              (Nonce N \<in> analz (Key`KK \<union> (knows Spy evs))) =
              (Nonce N \<in> analz (knows Spy evs)))"
apply (erule set_pur.induct)
apply (rule_tac [!] allI)+
apply (rule_tac [!] impI [THEN Nonce_analz_image_Key_lemma])+
apply (frule_tac [9] AuthReq_msg_in_analz_spies) \<comment>\<open>AReq\<close>
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (valid_certificate_tac [7]) \<comment>\<open>PReqUns\<close>
apply (simp_all
         del: image_insert image_Un imp_disjL
         add: analz_image_keys_simps disj_simps symKey_compromise
              analz_Key_image_insert_eq notin_image_iff
              analz_insert_simps analz_image_priEK)
  \<comment>\<open>8 seconds on a 1.6GHz machine\<close>
apply spy_analz \<comment>\<open>Fake\<close>
apply (blast elim!: ballE) \<comment>\<open>PReqS\<close>
done

lemma PANSecret_notin_spies:
     "[|Nonce (PANSecret k) \<in> analz (knows Spy evs);  evs \<in> set_pur|]
      ==> 
       (\<exists>V W X Y KC2 M. \<exists>P \<in> bad.
          Says (Cardholder k) M
               \<lbrace>\<lbrace>W, EXcrypt KC2 (pubEK P) X \<lbrace>Y, Nonce (PANSecret k)\<rbrace>\<rbrace>,
                 V\<rbrace>  \<in>  set evs)"
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_analz_spies)
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (simp_all
         del: image_insert image_Un imp_disjL
         add: analz_image_keys_simps disj_simps
              symKey_compromise pushes sign_def Nonce_compromise
              analz_Key_image_insert_eq notin_image_iff
              analz_insert_simps analz_image_priEK)
  \<comment>\<open>2.5 seconds on a 1.6GHz machine\<close>
apply spy_analz
apply (blast dest!: Gets_imp_knows_Spy [THEN analz.Inj])
apply (blast dest: Says_imp_knows_Spy [THEN analz.Inj] 
                   Gets_imp_knows_Spy [THEN analz.Inj])
apply (blast dest: Gets_imp_knows_Spy [THEN analz.Inj]) \<comment>\<open>PReqS\<close>
apply (blast dest: Says_imp_knows_Spy [THEN analz.Inj] 
                   Gets_imp_knows_Spy [THEN analz.Inj]) \<comment>\<open>PRes\<close>
done

text\<open>This theorem is a bit silly, in that many CardSecrets are 0!
  But then we don't care.  NOT USED\<close>
lemma CardSecret_notin_spies:
     "evs \<in> set_pur ==> Nonce (CardSecret i) \<notin> parts (knows Spy evs)"
by (erule set_pur.induct, auto)


subsection\<open>Confidentiality of PAN\<close>

lemma analz_image_pan_lemma:
     "(Pan P \<in> analz (Key`nE Un H)) --> (Pan P \<in> analz H)  ==>
      (Pan P \<in> analz (Key`nE Un H)) =   (Pan P \<in> analz H)"
by (blast intro: analz_mono [THEN [2] rev_subsetD])

text\<open>The \<open>(no_asm)\<close> attribute is essential, since it retains
  the quantifier and allows the simprule's condition to itself be simplified.\<close>
lemma analz_image_pan [rule_format (no_asm)]:
     "evs \<in> set_pur ==>
       \<forall>KK. (\<forall>K \<in> KK. K \<notin> range(\<lambda>C. priEK C)) -->
            (Pan P \<in> analz (Key`KK Un (knows Spy evs))) =
            (Pan P \<in> analz (knows Spy evs))"
apply (erule set_pur.induct)
apply (rule_tac [!] allI impI)+
apply (rule_tac [!] analz_image_pan_lemma)+
apply (frule_tac [9] AuthReq_msg_in_analz_spies) \<comment>\<open>AReq\<close>
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (valid_certificate_tac [7]) \<comment>\<open>PReqUns\<close>
apply (simp_all
         del: image_insert image_Un imp_disjL
         add: analz_image_keys_simps
              symKey_compromise pushes sign_def
              analz_Key_image_insert_eq notin_image_iff
              analz_insert_simps analz_image_priEK)
  \<comment>\<open>7 seconds on a 1.6GHz machine\<close>
apply spy_analz \<comment>\<open>Fake\<close>
apply auto
done

lemma analz_insert_pan:
     "[| evs \<in> set_pur;  K \<notin> range(\<lambda>C. priEK C) |] ==>
          (Pan P \<in> analz (insert (Key K) (knows Spy evs))) =
          (Pan P \<in> analz (knows Spy evs))"
by (simp del: image_insert image_Un
         add: analz_image_keys_simps analz_image_pan)

text\<open>Confidentiality of the PAN, unsigned case.\<close>
theorem pan_confidentiality_unsigned:
     "[| Pan (pan C) \<in> analz(knows Spy evs);  C = Cardholder k;
         CardSecret k = 0;  evs \<in> set_pur|]
    ==> \<exists>P M KC1 K X Y.
     Says C M \<lbrace>EXHcrypt KC1 (pubEK P) X (Pan (pan C)), Y\<rbrace>
          \<in> set evs  &
     P \<in> bad"
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_analz_spies) \<comment>\<open>AReq\<close>
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (valid_certificate_tac [7]) \<comment>\<open>PReqUns\<close>
apply (simp_all
         del: image_insert image_Un imp_disjL
         add: analz_image_keys_simps analz_insert_pan analz_image_pan
              notin_image_iff
              analz_insert_simps analz_image_priEK)
  \<comment>\<open>3 seconds on a 1.6GHz machine\<close>
apply spy_analz \<comment>\<open>Fake\<close>
apply blast \<comment>\<open>PReqUns: unsigned\<close>
apply force \<comment>\<open>PReqS: signed\<close>
done

text\<open>Confidentiality of the PAN, signed case.\<close>
theorem pan_confidentiality_signed:
 "[|Pan (pan C) \<in> analz(knows Spy evs);  C = Cardholder k;
    CardSecret k \<noteq> 0;  evs \<in> set_pur|]
  ==> \<exists>P M KC2 PIDualSign_1 PIDualSign_2 other OIDualSign.
      Says C M \<lbrace>\<lbrace>PIDualSign_1, 
                   EXcrypt KC2 (pubEK P) PIDualSign_2 \<lbrace>Pan (pan C), other\<rbrace>\<rbrace>, 
       OIDualSign\<rbrace> \<in> set evs  &  P \<in> bad"
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_analz_spies) \<comment>\<open>AReq\<close>
apply (valid_certificate_tac [8]) \<comment>\<open>PReqS\<close>
apply (valid_certificate_tac [7]) \<comment>\<open>PReqUns\<close>
apply (simp_all
         del: image_insert image_Un imp_disjL
         add: analz_image_keys_simps analz_insert_pan analz_image_pan
              notin_image_iff
              analz_insert_simps analz_image_priEK)
  \<comment>\<open>3 seconds on a 1.6GHz machine\<close>
apply spy_analz \<comment>\<open>Fake\<close>
apply force \<comment>\<open>PReqUns: unsigned\<close>
apply blast \<comment>\<open>PReqS: signed\<close>
done

text\<open>General goal: that C, M and PG agree on those details of the transaction
     that they are allowed to know about.  PG knows about price and account
     details.  M knows about the order description and price.  C knows both.\<close>


subsection\<open>Proofs Common to Signed and Unsigned Versions\<close>

lemma M_Notes_PG:
     "[|Notes M \<lbrace>Number LID_M, Agent P, Agent M, Agent C, etc\<rbrace> \<in> set evs;
        evs \<in> set_pur|] ==> \<exists>j. P = PG j"
by (erule rev_mp, erule set_pur.induct, simp_all)

text\<open>If we trust M, then @{term LID_M} determines his choice of P
      (Payment Gateway)\<close>
lemma goodM_gives_correct_PG:
     "[| MsgPInitRes = 
            \<lbrace>Number LID_M, xid, cc, cm, cert P EKj onlyEnc (priSK RCA)\<rbrace>;
         Crypt (priSK M) (Hash MsgPInitRes) \<in> parts (knows Spy evs);
         evs \<in> set_pur; M \<notin> bad |]
      ==> \<exists>j trans.
            P = PG j &
            Notes M \<lbrace>Number LID_M, Agent P, trans\<rbrace> \<in> set evs"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply (blast intro: M_Notes_PG)+
done

lemma C_gets_correct_PG:
     "[| Gets A (sign (priSK M) \<lbrace>Number LID_M, xid, cc, cm,
                              cert P EKj onlyEnc (priSK RCA)\<rbrace>) \<in> set evs;
         evs \<in> set_pur;  M \<notin> bad|]
      ==> \<exists>j trans.
            P = PG j &
            Notes M \<lbrace>Number LID_M, Agent P, trans\<rbrace> \<in> set evs &
            EKj = pubEK P"
by (rule refl [THEN goodM_gives_correct_PG, THEN exE], auto)

text\<open>When C receives PInitRes, he learns M's choice of P\<close>
lemma C_verifies_PInitRes:
 "[| MsgPInitRes = \<lbrace>Number LID_M, Number XID, Nonce Chall_C, Nonce Chall_M,
           cert P EKj onlyEnc (priSK RCA)\<rbrace>;
     Crypt (priSK M) (Hash MsgPInitRes) \<in> parts (knows Spy evs);
     evs \<in> set_pur;  M \<notin> bad|]
  ==> \<exists>j trans.
         Notes M \<lbrace>Number LID_M, Agent P, trans\<rbrace> \<in> set evs &
         P = PG j &
         EKj = pubEK P"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply (blast intro: M_Notes_PG)+
done

text\<open>Corollary of previous one\<close>
lemma Says_C_PInitRes:
     "[|Says A C (sign (priSK M)
                      \<lbrace>Number LID_M, Number XID,
                        Nonce Chall_C, Nonce Chall_M,
                        cert P EKj onlyEnc (priSK RCA)\<rbrace>)
           \<in> set evs;  M \<notin> bad;  evs \<in> set_pur|]
      ==> \<exists>j trans.
           Notes M \<lbrace>Number LID_M, Agent P, trans\<rbrace> \<in> set evs &
           P = PG j &
           EKj = pubEK (PG j)"
apply (frule Says_certificate_valid)
apply (auto simp add: sign_def)
apply (blast dest: refl [THEN goodM_gives_correct_PG])
apply (blast dest: refl [THEN C_verifies_PInitRes])
done

text\<open>When P receives an AuthReq, he knows that the signed part originated 
      with M. PIRes also has a signed message from M....\<close>
lemma P_verifies_AuthReq:
     "[| AuthReqData = \<lbrace>Number LID_M, Number XID, HOIData, HOD\<rbrace>;
         Crypt (priSK M) (Hash \<lbrace>AuthReqData, Hash P_I\<rbrace>)
           \<in> parts (knows Spy evs);
         evs \<in> set_pur;  M \<notin> bad|]
      ==> \<exists>j trans KM OIData HPIData.
            Notes M \<lbrace>Number LID_M, Agent (PG j), trans\<rbrace> \<in> set evs &
            Gets M \<lbrace>P_I, OIData, HPIData\<rbrace> \<in> set evs &
            Says M (PG j) (EncB (priSK M) KM (pubEK (PG j)) AuthReqData P_I)
              \<in> set evs"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all)
apply (frule_tac [4] M_Notes_PG, auto)
done

text\<open>When M receives AuthRes, he knows that P signed it, including
  the identifying tags and the purchase amount, which he can verify.
  (Although the spec has SIGNED and UNSIGNED forms of AuthRes, they
   send the same message to M.)  The conclusion is weak: M is existentially
  quantified! That is because Authorization Response does not refer to M, while
  the digital envelope weakens the link between @{term MsgAuthRes} and
  @{term"priSK M"}.  Changing the precondition to refer to 
  @{term "Crypt K (sign SK M)"} requires assuming @{term K} to be secure, since
  otherwise the Spy could create that message.\<close>
theorem M_verifies_AuthRes:
  "[| MsgAuthRes = \<lbrace>\<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>, 
                     Hash authCode\<rbrace>;
      Crypt (priSK (PG j)) (Hash MsgAuthRes) \<in> parts (knows Spy evs);
      PG j \<notin> bad;  evs \<in> set_pur|]
   ==> \<exists>M KM KP HOIData HOD P_I.
        Gets (PG j)
           (EncB (priSK M) KM (pubEK (PG j))
                    \<lbrace>Number LID_M, Number XID, HOIData, HOD\<rbrace>
                    P_I) \<in> set evs &
        Says (PG j) M
             (EncB (priSK (PG j)) KP (pubEK M)
              \<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>
              authCode) \<in> set evs"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply blast+
done


subsection\<open>Proofs for Unsigned Purchases\<close>

text\<open>What we can derive from the ASSUMPTION that C issued a purchase request.
   In the unsigned case, we must trust "C": there's no authentication.\<close>
lemma C_determines_EKj:
     "[| Says C M \<lbrace>EXHcrypt KC1 EKj \<lbrace>PIHead, Hash OIData\<rbrace> (Pan (pan C)),
                    OIData, Hash\<lbrace>PIHead, Pan (pan C)\<rbrace> \<rbrace> \<in> set evs;
         PIHead = \<lbrace>Number LID_M, Trans_details\<rbrace>;
         evs \<in> set_pur;  C = Cardholder k;  M \<notin> bad|]
  ==> \<exists>trans j.
               Notes M \<lbrace>Number LID_M, Agent (PG j), trans \<rbrace> \<in> set evs &
               EKj = pubEK (PG j)"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all)
apply (valid_certificate_tac [2]) \<comment>\<open>PReqUns\<close>
apply auto
apply (blast dest: Gets_imp_Says Says_C_PInitRes)
done


text\<open>Unicity of @{term LID_M} between Merchant and Cardholder notes\<close>
lemma unique_LID_M:
     "[|Notes (Merchant i) \<lbrace>Number LID_M, Agent P, Trans\<rbrace> \<in> set evs;
        Notes C \<lbrace>Number LID_M, Agent M, Agent C, Number OD,
             Number PA\<rbrace> \<in> set evs;
        evs \<in> set_pur|]
      ==> M = Merchant i & Trans = \<lbrace>Agent M, Agent C, Number OD, Number PA\<rbrace>"
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all)
apply (force dest!: Notes_imp_parts_subset_used)
done

text\<open>Unicity of @{term LID_M}, for two Merchant Notes events\<close>
lemma unique_LID_M2:
     "[|Notes M \<lbrace>Number LID_M, Trans\<rbrace> \<in> set evs;
        Notes M \<lbrace>Number LID_M, Trans'\<rbrace> \<in> set evs;
        evs \<in> set_pur|] ==> Trans' = Trans"
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all)
apply (force dest!: Notes_imp_parts_subset_used)
done

text\<open>Lemma needed below: for the case that
  if PRes is present, then @{term LID_M} has been used.\<close>
lemma signed_imp_used:
     "[| Crypt (priSK M) (Hash X) \<in> parts (knows Spy evs);
         M \<notin> bad;  evs \<in> set_pur|] ==> parts {X} \<subseteq> used evs"
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply safe
apply blast+
done

text\<open>Similar, with nested Hash\<close>
lemma signed_Hash_imp_used:
     "[| Crypt (priSK C) (Hash \<lbrace>H, Hash X\<rbrace>) \<in> parts (knows Spy evs);
         C \<notin> bad;  evs \<in> set_pur|] ==> parts {X} \<subseteq> used evs"
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply safe
apply blast+
done

text\<open>Lemma needed below: for the case that
  if PRes is present, then \<open>LID_M\<close> has been used.\<close>
lemma PRes_imp_LID_used:
     "[| Crypt (priSK M) (Hash \<lbrace>N, X\<rbrace>) \<in> parts (knows Spy evs);
         M \<notin> bad;  evs \<in> set_pur|] ==> N \<in> used evs"
by (drule signed_imp_used, auto)

text\<open>When C receives PRes, he knows that M and P agreed to the purchase details.
  He also knows that P is the same PG as before\<close>
lemma C_verifies_PRes_lemma:
     "[| Crypt (priSK M) (Hash MsgPRes) \<in> parts (knows Spy evs);
         Notes C \<lbrace>Number LID_M, Trans \<rbrace> \<in> set evs;
         Trans = \<lbrace> Agent M, Agent C, Number OrderDesc, Number PurchAmt \<rbrace>;
         MsgPRes = \<lbrace>Number LID_M, Number XID, Nonce Chall_C,
                Hash (Number PurchAmt)\<rbrace>;
         evs \<in> set_pur;  M \<notin> bad|]
  ==> \<exists>j KP.
        Notes M \<lbrace>Number LID_M, Agent (PG j), Trans \<rbrace>
          \<in> set evs &
        Gets M (EncB (priSK (PG j)) KP (pubEK M)
                \<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>
                authCode)
          \<in> set evs &
        Says M C (sign (priSK M) MsgPRes) \<in> set evs"
apply clarify
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply blast
apply blast
apply (blast dest: PRes_imp_LID_used)
apply (frule M_Notes_PG, auto)
apply (blast dest: unique_LID_M)
done

text\<open>When the Cardholder receives Purchase Response from an uncompromised
Merchant, he knows that M sent it. He also knows that M received a message signed
by a Payment Gateway chosen by M to authorize the purchase.\<close>
theorem C_verifies_PRes:
     "[| MsgPRes = \<lbrace>Number LID_M, Number XID, Nonce Chall_C,
                     Hash (Number PurchAmt)\<rbrace>;
         Gets C (sign (priSK M) MsgPRes) \<in> set evs;
         Notes C \<lbrace>Number LID_M, Agent M, Agent C, Number OrderDesc,
                   Number PurchAmt\<rbrace> \<in> set evs;
         evs \<in> set_pur;  M \<notin> bad|]
  ==> \<exists>P KP trans.
        Notes M \<lbrace>Number LID_M,Agent P, trans\<rbrace> \<in> set evs &
        Gets M (EncB (priSK P) KP (pubEK M)
                \<lbrace>Number LID_M, Number XID, Number PurchAmt\<rbrace>
                authCode)  \<in>  set evs &
        Says M C (sign (priSK M) MsgPRes) \<in> set evs"
apply (rule C_verifies_PRes_lemma [THEN exE])
apply (auto simp add: sign_def)
done

subsection\<open>Proofs for Signed Purchases\<close>

text\<open>Some Useful Lemmas: the cardholder knows what he is doing\<close>

lemma Crypt_imp_Says_Cardholder:
     "[| Crypt K \<lbrace>\<lbrace>\<lbrace>Number LID_M, others\<rbrace>, Hash OIData\<rbrace>, Hash PANData\<rbrace>
           \<in> parts (knows Spy evs);
         PANData = \<lbrace>Pan (pan (Cardholder k)), Nonce (PANSecret k)\<rbrace>;
         Key K \<notin> analz (knows Spy evs);
         evs \<in> set_pur|]
  ==> \<exists>M shash EK HPIData.
       Says (Cardholder k) M \<lbrace>\<lbrace>shash,
          Crypt K
            \<lbrace>\<lbrace>\<lbrace>Number LID_M, others\<rbrace>, Hash OIData\<rbrace>, Hash PANData\<rbrace>,
           Crypt EK \<lbrace>Key K, PANData\<rbrace>\<rbrace>,
          OIData, HPIData\<rbrace> \<in> set evs"
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct, analz_mono_contra)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply auto
done

lemma Says_PReqS_imp_trans_details_C:
     "[| MsgPReqS = \<lbrace>\<lbrace>shash,
                 Crypt K
                  \<lbrace>\<lbrace>\<lbrace>Number LID_M, PIrest\<rbrace>, Hash OIData\<rbrace>, hashpd\<rbrace>,
            cryptek\<rbrace>, data\<rbrace>;
         Says (Cardholder k) M MsgPReqS \<in> set evs;
         evs \<in> set_pur |]
   ==> \<exists>trans.
           Notes (Cardholder k) 
                 \<lbrace>Number LID_M, Agent M, Agent (Cardholder k), trans\<rbrace>
            \<in> set evs"
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (simp_all (no_asm_simp))
apply auto
done

text\<open>Can't happen: only Merchants create this type of Note\<close>
lemma Notes_Cardholder_self_False:
     "[|Notes (Cardholder k)
          \<lbrace>Number n, Agent P, Agent (Cardholder k), Agent C, etc\<rbrace> \<in> set evs;
        evs \<in> set_pur|] ==> False"
by (erule rev_mp, erule set_pur.induct, auto)

text\<open>When M sees a dual signature, he knows that it originated with C.
  Using XID he knows it was intended for him.
  This guarantee isn't useful to P, who never gets OIData.\<close>
theorem M_verifies_Signed_PReq:
 "[| MsgDualSign = \<lbrace>HPIData, Hash OIData\<rbrace>;
     OIData = \<lbrace>Number LID_M, etc\<rbrace>;
     Crypt (priSK C) (Hash MsgDualSign) \<in> parts (knows Spy evs);
     Notes M \<lbrace>Number LID_M, Agent P, extras\<rbrace> \<in> set evs;
     M = Merchant i;  C = Cardholder k;  C \<notin> bad;  evs \<in> set_pur|]
  ==> \<exists>PIData PICrypt.
        HPIData = Hash PIData &
        Says C M \<lbrace>\<lbrace>sign (priSK C) MsgDualSign, PICrypt\<rbrace>, OIData, Hash PIData\<rbrace>
          \<in> set evs"
apply clarify
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct)
apply (frule_tac [9] AuthReq_msg_in_parts_spies) \<comment>\<open>AuthReq\<close>
apply simp_all
apply blast
apply (metis subsetD insert_subset parts.Fst parts_increasing signed_Hash_imp_used)
apply (metis unique_LID_M)
apply (blast dest!: Notes_Cardholder_self_False)
done

text\<open>When P sees a dual signature, he knows that it originated with C.
  and was intended for M. This guarantee isn't useful to M, who never gets
  PIData. I don't see how to link @{term "PG j"} and \<open>LID_M\<close> without
  assuming @{term "M \<notin> bad"}.\<close>
theorem P_verifies_Signed_PReq:
     "[| MsgDualSign = \<lbrace>Hash PIData, HOIData\<rbrace>;
         PIData = \<lbrace>PIHead, PANData\<rbrace>;
         PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M,
                    TransStain\<rbrace>;
         Crypt (priSK C) (Hash MsgDualSign) \<in> parts (knows Spy evs);
         evs \<in> set_pur;  C \<notin> bad;  M \<notin> bad|]
    ==> \<exists>OIData OrderDesc K j trans.
          HOD = Hash\<lbrace>Number OrderDesc, Number PurchAmt\<rbrace> &
          HOIData = Hash OIData &
          Notes M \<lbrace>Number LID_M, Agent (PG j), trans\<rbrace> \<in> set evs &
          Says C M \<lbrace>\<lbrace>sign (priSK C) MsgDualSign,
                     EXcrypt K (pubEK (PG j))
                                \<lbrace>PIHead, Hash OIData\<rbrace> PANData\<rbrace>,
                     OIData, Hash PIData\<rbrace>
            \<in> set evs"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all)
apply (auto dest!: C_gets_correct_PG)
done

lemma C_determines_EKj_signed:
     "[| Says C M \<lbrace>\<lbrace>sign (priSK C) text,
                      EXcrypt K EKj \<lbrace>PIHead, X\<rbrace> Y\<rbrace>, Z\<rbrace> \<in> set evs;
         PIHead = \<lbrace>Number LID_M, Number XID, W\<rbrace>;
         C = Cardholder k;  evs \<in> set_pur;  M \<notin> bad|]
  ==> \<exists> trans j.
         Notes M \<lbrace>Number LID_M, Agent (PG j), trans\<rbrace> \<in> set evs &
         EKj = pubEK (PG j)"
apply clarify
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all, auto)
apply (blast dest: C_gets_correct_PG)
done

lemma M_Says_AuthReq:
     "[| AuthReqData = \<lbrace>Number LID_M, Number XID, HOIData, HOD\<rbrace>;
         sign (priSK M) \<lbrace>AuthReqData, Hash P_I\<rbrace> \<in> parts (knows Spy evs);
         evs \<in> set_pur;  M \<notin> bad|]
   ==> \<exists>j trans KM.
           Notes M \<lbrace>Number LID_M, Agent (PG j), trans \<rbrace> \<in> set evs &
             Says M (PG j)
               (EncB (priSK M) KM (pubEK (PG j)) AuthReqData P_I)
              \<in> set evs"
apply (rule refl [THEN P_verifies_AuthReq, THEN exE])
apply (auto simp add: sign_def)
done

text\<open>A variant of \<open>M_verifies_Signed_PReq\<close> with explicit PI information.
  Even here we cannot be certain about what C sent to M, since a bad
  PG could have replaced the two key fields.  (NOT USED)\<close>
lemma Signed_PReq_imp_Says_Cardholder:
     "[| MsgDualSign = \<lbrace>Hash PIData, Hash OIData\<rbrace>;
         OIData = \<lbrace>Number LID_M, Number XID, Nonce Chall_C, HOD, etc\<rbrace>;
         PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M,
                    TransStain\<rbrace>;
         PIData = \<lbrace>PIHead, PANData\<rbrace>;
         Crypt (priSK C) (Hash MsgDualSign) \<in> parts (knows Spy evs);
         M = Merchant i;  C = Cardholder k;  C \<notin> bad;  evs \<in> set_pur|]
      ==> \<exists>KC EKj.
            Says C M \<lbrace>\<lbrace>sign (priSK C) MsgDualSign,
                       EXcrypt KC EKj \<lbrace>PIHead, Hash OIData\<rbrace> PANData\<rbrace>,
                       OIData, Hash PIData\<rbrace>
              \<in> set evs"
apply clarify
apply hypsubst_thin
apply (erule rev_mp)
apply (erule rev_mp)
apply (erule set_pur.induct, simp_all, auto)
done

text\<open>When P receives an AuthReq and a dual signature, he knows that C and M
  agree on the essential details.  PurchAmt however is never sent by M to
  P; instead C and M both send 
     @{term "HOD = Hash\<lbrace>Number OrderDesc, Number PurchAmt\<rbrace>"}
  and P compares the two copies of HOD.

  Agreement can't be proved for some things, including the symmetric keys
  used in the digital envelopes.  On the other hand, M knows the true identity
  of PG (namely j'), and sends AReq there; he can't, however, check that
  the EXcrypt involves the correct PG's key.
\<close>
theorem P_sees_CM_agreement:
     "[| AuthReqData = \<lbrace>Number LID_M, Number XID, HOIData, HOD\<rbrace>;
         KC \<in> symKeys;
         Gets (PG j) (EncB (priSK M) KM (pubEK (PG j)) AuthReqData P_I)
           \<in> set evs;
         C = Cardholder k;
         PI_sign = sign (priSK C) \<lbrace>Hash PIData, HOIData\<rbrace>;
         P_I = \<lbrace>PI_sign,
                 EXcrypt KC (pubEK (PG j)) \<lbrace>PIHead, HOIData\<rbrace> PANData\<rbrace>;
         PANData = \<lbrace>Pan (pan C), Nonce (PANSecret k)\<rbrace>;
         PIData = \<lbrace>PIHead, PANData\<rbrace>;
         PIHead = \<lbrace>Number LID_M, Number XID, HOD, Number PurchAmt, Agent M,
                    TransStain\<rbrace>;
         evs \<in> set_pur;  C \<notin> bad;  M \<notin> bad|]
  ==> \<exists>OIData OrderDesc KM' trans j' KC' KC'' P_I' P_I''.
           HOD = Hash\<lbrace>Number OrderDesc, Number PurchAmt\<rbrace> &
           HOIData = Hash OIData &
           Notes M \<lbrace>Number LID_M, Agent (PG j'), trans\<rbrace> \<in> set evs &
           Says C M \<lbrace>P_I', OIData, Hash PIData\<rbrace> \<in> set evs &
           Says M (PG j') (EncB (priSK M) KM' (pubEK (PG j'))
                           AuthReqData P_I'')  \<in>  set evs &
           P_I' = \<lbrace>PI_sign,
             EXcrypt KC' (pubEK (PG j')) \<lbrace>PIHead, Hash OIData\<rbrace> PANData\<rbrace> &
           P_I'' = \<lbrace>PI_sign,
             EXcrypt KC'' (pubEK (PG j)) \<lbrace>PIHead, Hash OIData\<rbrace> PANData\<rbrace>"
apply clarify
apply (rule exE)
apply (rule P_verifies_Signed_PReq [OF refl refl refl])
apply (simp (no_asm_use) add: sign_def EncB_def, blast)
apply (assumption+, clarify, simp)
apply (drule Gets_imp_knows_Spy [THEN parts.Inj], assumption)
apply (blast elim: EncB_partsE dest: refl [THEN M_Says_AuthReq] unique_LID_M2)
done

end
