(*  Title:      HOL/Integ/int_arith1.ML
    ID:         $Id$
    Authors:    Larry Paulson and Tobias Nipkow

Simprocs and decision procedure for linear arithmetic.
*)

(** Misc ML bindings **)

val bin_succ_Pls = thm"bin_succ_Pls";
val bin_succ_Min = thm"bin_succ_Min";
val bin_succ_1 = thm"bin_succ_1";
val bin_succ_0 = thm"bin_succ_0";

val bin_pred_Pls = thm"bin_pred_Pls";
val bin_pred_Min = thm"bin_pred_Min";
val bin_pred_1 = thm"bin_pred_1";
val bin_pred_0 = thm"bin_pred_0";

val bin_minus_Pls = thm"bin_minus_Pls";
val bin_minus_Min = thm"bin_minus_Min";
val bin_minus_1 = thm"bin_minus_1";
val bin_minus_0 = thm"bin_minus_0";

val bin_add_Pls = thm"bin_add_Pls";
val bin_add_Min = thm"bin_add_Min";
val bin_add_BIT_11 = thm"bin_add_BIT_11";
val bin_add_BIT_10 = thm"bin_add_BIT_10";
val bin_add_BIT_0 = thm"bin_add_BIT_0";
val bin_add_Pls_right = thm"bin_add_Pls_right";
val bin_add_Min_right = thm"bin_add_Min_right";

val bin_mult_Pls = thm"bin_mult_Pls";
val bin_mult_Min = thm"bin_mult_Min";
val bin_mult_1 = thm"bin_mult_1";
val bin_mult_0 = thm"bin_mult_0";

val neg_def = thm "neg_def";
val iszero_def = thm "iszero_def";

val number_of_succ = thm"number_of_succ";
val number_of_pred = thm"number_of_pred";
val number_of_minus = thm"number_of_minus";
val number_of_add = thm"number_of_add";
val diff_number_of_eq = thm"diff_number_of_eq";
val number_of_mult = thm"number_of_mult";
val double_number_of_BIT = thm"double_number_of_BIT";
val numeral_0_eq_0 = thm"numeral_0_eq_0";
val numeral_1_eq_1 = thm"numeral_1_eq_1";
val numeral_m1_eq_minus_1 = thm"numeral_m1_eq_minus_1";
val mult_minus1 = thm"mult_minus1";
val mult_minus1_right = thm"mult_minus1_right";
val minus_number_of_mult = thm"minus_number_of_mult";
val zero_less_nat_eq = thm"zero_less_nat_eq";
val eq_number_of_eq = thm"eq_number_of_eq";
val iszero_number_of_Pls = thm"iszero_number_of_Pls";
val nonzero_number_of_Min = thm"nonzero_number_of_Min";
val iszero_number_of_BIT = thm"iszero_number_of_BIT";
val iszero_number_of_0 = thm"iszero_number_of_0";
val iszero_number_of_1 = thm"iszero_number_of_1";
val less_number_of_eq_neg = thm"less_number_of_eq_neg";
val le_number_of_eq = thm"le_number_of_eq";
val not_neg_number_of_Pls = thm"not_neg_number_of_Pls";
val neg_number_of_Min = thm"neg_number_of_Min";
val neg_number_of_BIT = thm"neg_number_of_BIT";
val le_number_of_eq_not_less = thm"le_number_of_eq_not_less";
val abs_number_of = thm"abs_number_of";
val number_of_reorient = thm"number_of_reorient";
val add_number_of_left = thm"add_number_of_left";
val mult_number_of_left = thm"mult_number_of_left";
val add_number_of_diff1 = thm"add_number_of_diff1";
val add_number_of_diff2 = thm"add_number_of_diff2";
val less_iff_diff_less_0 = thm"less_iff_diff_less_0";
val eq_iff_diff_eq_0 = thm"eq_iff_diff_eq_0";
val le_iff_diff_le_0 = thm"le_iff_diff_le_0";

val bin_arith_extra_simps = thms"bin_arith_extra_simps";
val bin_arith_simps = thms"bin_arith_simps";
val bin_rel_simps = thms"bin_rel_simps";

val zless_imp_add1_zle = thm "zless_imp_add1_zle";

val combine_common_factor = thm"combine_common_factor";
val eq_add_iff1 = thm"eq_add_iff1";
val eq_add_iff2 = thm"eq_add_iff2";
val less_add_iff1 = thm"less_add_iff1";
val less_add_iff2 = thm"less_add_iff2";
val le_add_iff1 = thm"le_add_iff1";
val le_add_iff2 = thm"le_add_iff2";

val arith_special = thms"arith_special";

structure Bin_Simprocs =
  struct
  fun prove_conv tacs sg (hyps: thm list) xs (t, u) =
    if t aconv u then NONE
    else
      let val eq = HOLogic.mk_Trueprop (HOLogic.mk_eq (t, u))
      in SOME (Tactic.prove sg xs [] eq (K (EVERY tacs))) end

  fun prove_conv_nohyps tacs sg = prove_conv tacs sg [];
  fun prove_conv_nohyps_novars tacs sg = prove_conv tacs sg [] [];

  fun prep_simproc (name, pats, proc) =
    Simplifier.simproc (Theory.sign_of (the_context())) name pats proc;

  fun is_numeral (Const("Numeral.number_of", _) $ w) = true
    | is_numeral _ = false

  fun simplify_meta_eq f_number_of_eq f_eq =
      mk_meta_eq ([f_eq, f_number_of_eq] MRS trans)

  (*reorientation simprules using ==, for the following simproc*)
  val meta_zero_reorient = zero_reorient RS eq_reflection
  val meta_one_reorient = one_reorient RS eq_reflection
  val meta_number_of_reorient = number_of_reorient RS eq_reflection

  (*reorientation simplification procedure: reorients (polymorphic) 
    0 = x, 1 = x, nnn = x provided x isn't 0, 1 or a numeral.*)
  fun reorient_proc sg _ (_ $ t $ u) =
    case u of
	Const("0", _) => NONE
      | Const("1", _) => NONE
      | Const("Numeral.number_of", _) $ _ => NONE
      | _ => SOME (case t of
		  Const("0", _) => meta_zero_reorient
		| Const("1", _) => meta_one_reorient
		| Const("Numeral.number_of", _) $ _ => meta_number_of_reorient)

  val reorient_simproc = 
      prep_simproc ("reorient_simproc", ["0=x", "1=x", "number_of w = x"], reorient_proc)

  end;


Addsimps arith_special;
Addsimprocs [Bin_Simprocs.reorient_simproc];


structure Int_Numeral_Simprocs =
struct

(*Maps 0 to Numeral0 and 1 to Numeral1 so that arithmetic in simprocs
  isn't complicated by the abstract 0 and 1.*)
val numeral_syms = [numeral_0_eq_0 RS sym, numeral_1_eq_1 RS sym];

(** New term ordering so that AC-rewriting brings numerals to the front **)

(*Order integers by absolute value and then by sign. The standard integer
  ordering is not well-founded.*)
fun num_ord (i,j) =
      (case IntInf.compare (IntInf.abs i, IntInf.abs j) of
            EQUAL => Int.compare (IntInf.sign i, IntInf.sign j) 
          | ord => ord);

(*This resembles Term.term_ord, but it puts binary numerals before other
  non-atomic terms.*)
local open Term 
in 
fun numterm_ord (Abs (_, T, t), Abs(_, U, u)) =
      (case numterm_ord (t, u) of EQUAL => typ_ord (T, U) | ord => ord)
  | numterm_ord
     (Const("Numeral.number_of", _) $ v, Const("Numeral.number_of", _) $ w) =
     num_ord (HOLogic.dest_binum v, HOLogic.dest_binum w)
  | numterm_ord (Const("Numeral.number_of", _) $ _, _) = LESS
  | numterm_ord (_, Const("Numeral.number_of", _) $ _) = GREATER
  | numterm_ord (t, u) =
      (case Int.compare (size_of_term t, size_of_term u) of
        EQUAL =>
          let val (f, ts) = strip_comb t and (g, us) = strip_comb u in
            (case hd_ord (f, g) of EQUAL => numterms_ord (ts, us) | ord => ord)
          end
      | ord => ord)
and numterms_ord (ts, us) = list_ord numterm_ord (ts, us)
end;

fun numtermless tu = (numterm_ord tu = LESS);

(*Defined in this file, but perhaps needed only for simprocs of type nat.*)
val num_ss = HOL_ss settermless numtermless;


(** Utilities **)

fun mk_numeral T n = HOLogic.number_of_const T $ HOLogic.mk_bin n;

(*Decodes a binary INTEGER*)
fun dest_numeral (Const("0", _)) = 0
  | dest_numeral (Const("1", _)) = 1
  | dest_numeral (Const("Numeral.number_of", _) $ w) =
     (HOLogic.dest_binum w
      handle TERM _ => raise TERM("Int_Numeral_Simprocs.dest_numeral:1", [w]))
  | dest_numeral t = raise TERM("Int_Numeral_Simprocs.dest_numeral:2", [t]);

fun find_first_numeral past (t::terms) =
        ((dest_numeral t, rev past @ terms)
         handle TERM _ => find_first_numeral (t::past) terms)
  | find_first_numeral past [] = raise TERM("find_first_numeral", []);

val mk_plus = HOLogic.mk_binop "op +";

fun mk_minus t = 
  let val T = Term.fastype_of t
  in Const ("uminus", T --> T) $ t
  end;

(*Thus mk_sum[t] yields t+0; longer sums don't have a trailing zero*)
fun mk_sum T []        = mk_numeral T 0
  | mk_sum T [t,u]     = mk_plus (t, u)
  | mk_sum T (t :: ts) = mk_plus (t, mk_sum T ts);

(*this version ALWAYS includes a trailing zero*)
fun long_mk_sum T []        = mk_numeral T 0
  | long_mk_sum T (t :: ts) = mk_plus (t, mk_sum T ts);

val dest_plus = HOLogic.dest_bin "op +" Term.dummyT;

(*decompose additions AND subtractions as a sum*)
fun dest_summing (pos, Const ("op +", _) $ t $ u, ts) =
        dest_summing (pos, t, dest_summing (pos, u, ts))
  | dest_summing (pos, Const ("op -", _) $ t $ u, ts) =
        dest_summing (pos, t, dest_summing (not pos, u, ts))
  | dest_summing (pos, t, ts) =
        if pos then t::ts else mk_minus t :: ts;

fun dest_sum t = dest_summing (true, t, []);

val mk_diff = HOLogic.mk_binop "op -";
val dest_diff = HOLogic.dest_bin "op -" Term.dummyT;

val mk_times = HOLogic.mk_binop "op *";

fun mk_prod T = 
  let val one = mk_numeral T 1
  fun mk [] = one
    | mk [t] = t
    | mk (t :: ts) = if t = one then mk ts else mk_times (t, mk ts)
  in mk end;

(*This version ALWAYS includes a trailing one*)
fun long_mk_prod T []        = mk_numeral T 1
  | long_mk_prod T (t :: ts) = mk_times (t, mk_prod T ts);

val dest_times = HOLogic.dest_bin "op *" Term.dummyT;

fun dest_prod t =
      let val (t,u) = dest_times t
      in  dest_prod t @ dest_prod u  end
      handle TERM _ => [t];

(*DON'T do the obvious simplifications; that would create special cases*)
fun mk_coeff (k, t) = mk_times (mk_numeral (Term.fastype_of t) k, t);

(*Express t as a product of (possibly) a numeral with other sorted terms*)
fun dest_coeff sign (Const ("uminus", _) $ t) = dest_coeff (~sign) t
  | dest_coeff sign t =
    let val ts = sort Term.term_ord (dest_prod t)
        val (n, ts') = find_first_numeral [] ts
                          handle TERM _ => (1, ts)
    in (sign*n, mk_prod (Term.fastype_of t) ts') end;

(*Find first coefficient-term THAT MATCHES u*)
fun find_first_coeff past u [] = raise TERM("find_first_coeff", [])
  | find_first_coeff past u (t::terms) =
        let val (n,u') = dest_coeff 1 t
        in  if u aconv u' then (n, rev past @ terms)
                          else find_first_coeff (t::past) u terms
        end
        handle TERM _ => find_first_coeff (t::past) u terms;


(*Simplify Numeral0+n, n+Numeral0, Numeral1*n, n*Numeral1*)
val add_0s =  thms "add_0s";
val mult_1s = thms "mult_1s";

(*To perform binary arithmetic.  The "left" rewriting handles patterns
  created by the simprocs, such as 3 * (5 * x). *)
val bin_simps = [numeral_0_eq_0 RS sym, numeral_1_eq_1 RS sym,
                 add_number_of_left, mult_number_of_left] @
                bin_arith_simps @ bin_rel_simps;

(*Binary arithmetic BUT NOT ADDITION since it may collapse adjacent terms
  during re-arrangement*)
val non_add_bin_simps = 
    bin_simps \\ [add_number_of_left, number_of_add RS sym];

(*To evaluate binary negations of coefficients*)
val minus_simps = [numeral_m1_eq_minus_1 RS sym, number_of_minus RS sym,
                   bin_minus_1, bin_minus_0, bin_minus_Pls, bin_minus_Min,
                   bin_pred_1, bin_pred_0, bin_pred_Pls, bin_pred_Min];

(*To let us treat subtraction as addition*)
val diff_simps = [diff_minus, minus_add_distrib, minus_minus];

(*push the unary minus down: - x * y = x * - y *)
val minus_mult_eq_1_to_2 =
    [minus_mult_left RS sym, minus_mult_right] MRS trans |> standard;

(*to extract again any uncancelled minuses*)
val minus_from_mult_simps =
    [minus_minus, minus_mult_left RS sym, minus_mult_right RS sym];

(*combine unary minus with numeric literals, however nested within a product*)
val mult_minus_simps =
    [mult_assoc, minus_mult_left, minus_mult_eq_1_to_2];

(*Apply the given rewrite (if present) just once*)
fun trans_tac NONE      = all_tac
  | trans_tac (SOME th) = ALLGOALS (rtac (th RS trans));

fun simplify_meta_eq rules =
    simplify (HOL_basic_ss addeqcongs[eq_cong2] addsimps rules)
    o mk_meta_eq;

structure CancelNumeralsCommon =
  struct
  val mk_sum            = mk_sum
  val dest_sum          = dest_sum
  val mk_coeff          = mk_coeff
  val dest_coeff        = dest_coeff 1
  val find_first_coeff  = find_first_coeff []
  val trans_tac         = trans_tac
  val norm_tac =
     ALLGOALS (simp_tac (num_ss addsimps numeral_syms@add_0s@mult_1s@
                                         diff_simps@minus_simps@add_ac))
     THEN ALLGOALS (simp_tac (num_ss addsimps non_add_bin_simps@mult_minus_simps))
     THEN ALLGOALS (simp_tac (num_ss addsimps minus_from_mult_simps@
                                              add_ac@mult_ac))
  val numeral_simp_tac  = ALLGOALS (simp_tac (HOL_ss addsimps add_0s@bin_simps))
  val simplify_meta_eq  = simplify_meta_eq (add_0s@mult_1s)
  end;


structure EqCancelNumerals = CancelNumeralsFun
 (open CancelNumeralsCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_eq
  val dest_bal = HOLogic.dest_bin "op =" Term.dummyT
  val bal_add1 = eq_add_iff1 RS trans
  val bal_add2 = eq_add_iff2 RS trans
);

structure LessCancelNumerals = CancelNumeralsFun
 (open CancelNumeralsCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_binrel "op <"
  val dest_bal = HOLogic.dest_bin "op <" Term.dummyT
  val bal_add1 = less_add_iff1 RS trans
  val bal_add2 = less_add_iff2 RS trans
);

structure LeCancelNumerals = CancelNumeralsFun
 (open CancelNumeralsCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_binrel "op <="
  val dest_bal = HOLogic.dest_bin "op <=" Term.dummyT
  val bal_add1 = le_add_iff1 RS trans
  val bal_add2 = le_add_iff2 RS trans
);

val cancel_numerals =
  map Bin_Simprocs.prep_simproc
   [("inteq_cancel_numerals",
     ["(l::'a::number_ring) + m = n",
      "(l::'a::number_ring) = m + n",
      "(l::'a::number_ring) - m = n",
      "(l::'a::number_ring) = m - n",
      "(l::'a::number_ring) * m = n",
      "(l::'a::number_ring) = m * n"],
     EqCancelNumerals.proc),
    ("intless_cancel_numerals",
     ["(l::'a::{ordered_idom,number_ring}) + m < n",
      "(l::'a::{ordered_idom,number_ring}) < m + n",
      "(l::'a::{ordered_idom,number_ring}) - m < n",
      "(l::'a::{ordered_idom,number_ring}) < m - n",
      "(l::'a::{ordered_idom,number_ring}) * m < n",
      "(l::'a::{ordered_idom,number_ring}) < m * n"],
     LessCancelNumerals.proc),
    ("intle_cancel_numerals",
     ["(l::'a::{ordered_idom,number_ring}) + m <= n",
      "(l::'a::{ordered_idom,number_ring}) <= m + n",
      "(l::'a::{ordered_idom,number_ring}) - m <= n",
      "(l::'a::{ordered_idom,number_ring}) <= m - n",
      "(l::'a::{ordered_idom,number_ring}) * m <= n",
      "(l::'a::{ordered_idom,number_ring}) <= m * n"],
     LeCancelNumerals.proc)];


structure CombineNumeralsData =
  struct
  val add               = IntInf.+ 
  val mk_sum            = long_mk_sum    (*to work for e.g. 2*x + 3*x *)
  val dest_sum          = dest_sum
  val mk_coeff          = mk_coeff
  val dest_coeff        = dest_coeff 1
  val left_distrib      = combine_common_factor RS trans
  val prove_conv        = Bin_Simprocs.prove_conv_nohyps
  val trans_tac          = trans_tac
  val norm_tac =
     ALLGOALS (simp_tac (num_ss addsimps numeral_syms@add_0s@mult_1s@
                                         diff_simps@minus_simps@add_ac))
     THEN ALLGOALS (simp_tac (num_ss addsimps non_add_bin_simps@mult_minus_simps))
     THEN ALLGOALS (simp_tac (num_ss addsimps minus_from_mult_simps@
                                              add_ac@mult_ac))
  val numeral_simp_tac  = ALLGOALS
                    (simp_tac (HOL_ss addsimps add_0s@bin_simps))
  val simplify_meta_eq  = simplify_meta_eq (add_0s@mult_1s)
  end;

structure CombineNumerals = CombineNumeralsFun(CombineNumeralsData);

val combine_numerals =
  Bin_Simprocs.prep_simproc
    ("int_combine_numerals", 
     ["(i::'a::number_ring) + j", "(i::'a::number_ring) - j"], 
     CombineNumerals.proc);

end;

Addsimprocs Int_Numeral_Simprocs.cancel_numerals;
Addsimprocs [Int_Numeral_Simprocs.combine_numerals];

(*examples:
print_depth 22;
set timing;
set trace_simp;
fun test s = (Goal s, by (Simp_tac 1));

test "l + 2 + 2 + 2 + (l + 2) + (oo + 2) = (uu::int)";

test "2*u = (u::int)";
test "(i + j + 12 + (k::int)) - 15 = y";
test "(i + j + 12 + (k::int)) - 5 = y";

test "y - b < (b::int)";
test "y - (3*b + c) < (b::int) - 2*c";

test "(2*x - (u*v) + y) - v*3*u = (w::int)";
test "(2*x*u*v + (u*v)*4 + y) - v*u*4 = (w::int)";
test "(2*x*u*v + (u*v)*4 + y) - v*u = (w::int)";
test "u*v - (x*u*v + (u*v)*4 + y) = (w::int)";

test "(i + j + 12 + (k::int)) = u + 15 + y";
test "(i + j*2 + 12 + (k::int)) = j + 5 + y";

test "2*y + 3*z + 6*w + 2*y + 3*z + 2*u = 2*y' + 3*z' + 6*w' + 2*y' + 3*z' + u + (vv::int)";

test "a + -(b+c) + b = (d::int)";
test "a + -(b+c) - b = (d::int)";

(*negative numerals*)
test "(i + j + -2 + (k::int)) - (u + 5 + y) = zz";
test "(i + j + -3 + (k::int)) < u + 5 + y";
test "(i + j + 3 + (k::int)) < u + -6 + y";
test "(i + j + -12 + (k::int)) - 15 = y";
test "(i + j + 12 + (k::int)) - -15 = y";
test "(i + j + -12 + (k::int)) - -15 = y";
*)


(** Constant folding for multiplication in semirings **)

(*We do not need folding for addition: combine_numerals does the same thing*)

structure Semiring_Times_Assoc_Data : ASSOC_FOLD_DATA =
struct
  val ss                = HOL_ss
  val eq_reflection     = eq_reflection
  val sg_ref = Sign.self_ref (Theory.sign_of (the_context ()))
  val add_ac = mult_ac
end;

structure Semiring_Times_Assoc = Assoc_Fold (Semiring_Times_Assoc_Data);

val assoc_fold_simproc =
  Bin_Simprocs.prep_simproc
   ("semiring_assoc_fold", ["(a::'a::comm_semiring_1_cancel) * b"],
    Semiring_Times_Assoc.proc);

Addsimprocs [assoc_fold_simproc];




(*** decision procedure for linear arithmetic ***)

(*---------------------------------------------------------------------------*)
(* Linear arithmetic                                                         *)
(*---------------------------------------------------------------------------*)

(*
Instantiation of the generic linear arithmetic package for int.
*)

(* Update parameters of arithmetic prover *)
local

(* reduce contradictory <= to False *)
val add_rules =
    simp_thms @ bin_arith_simps @ bin_rel_simps @ arith_special @
    [neg_le_iff_le, numeral_0_eq_0, numeral_1_eq_1,
     minus_zero, diff_minus, left_minus, right_minus,
     mult_zero_left, mult_zero_right, mult_1, mult_1_right,
     minus_mult_left RS sym, minus_mult_right RS sym,
     minus_add_distrib, minus_minus, mult_assoc,
     of_nat_0, of_nat_1, of_nat_Suc, of_nat_add, of_nat_mult,
     of_int_0, of_int_1, of_int_add, of_int_mult, int_eq_of_nat,
     zero_neq_one, zero_less_one, zero_le_one, 
     zero_neq_one RS not_sym, not_one_le_zero, not_one_less_zero];

val simprocs = [assoc_fold_simproc, Int_Numeral_Simprocs.combine_numerals]@
               Int_Numeral_Simprocs.cancel_numerals;

in

val int_arith_setup =
 [Fast_Arith.map_data (fn {add_mono_thms, mult_mono_thms, inj_thms, lessD, neqE, simpset} =>
   {add_mono_thms = add_mono_thms,
    mult_mono_thms = [mult_strict_left_mono,mult_left_mono] @ mult_mono_thms,
    inj_thms = [zle_int RS iffD2,int_int_eq RS iffD2] @ inj_thms,
    lessD = lessD @ [zless_imp_add1_zle],
    neqE = thm "linorder_neqE_int" :: neqE,
    simpset = simpset addsimps add_rules
                      addsimprocs simprocs
                      addcongs [if_weak_cong]}),
  arith_inj_const ("IntDef.of_nat", HOLogic.natT --> HOLogic.intT),
  arith_inj_const ("IntDef.int", HOLogic.natT --> HOLogic.intT),
  arith_discrete "IntDef.int"];

end;

val fast_int_arith_simproc =
  Simplifier.simproc (Theory.sign_of (the_context()))
  "fast_int_arith" 
     ["(m::'a::{ordered_idom,number_ring}) < n",
      "(m::'a::{ordered_idom,number_ring}) <= n",
      "(m::'a::{ordered_idom,number_ring}) = n"] Fast_Arith.lin_arith_prover;

Addsimprocs [fast_int_arith_simproc]


(* Some test data
Goal "!!a::int. [| a <= b; c <= d; x+y<z |] ==> a+c <= b+d";
by (fast_arith_tac 1);
Goal "!!a::int. [| a < b; c < d |] ==> a-d+ 2 <= b+(-c)";
by (fast_arith_tac 1);
Goal "!!a::int. [| a < b; c < d |] ==> a+c+ 1 < b+d";
by (fast_arith_tac 1);
Goal "!!a::int. [| a <= b; b+b <= c |] ==> a+a <= c";
by (fast_arith_tac 1);
Goal "!!a::int. [| a+b <= i+j; a<=b; i<=j |] \
\     ==> a+a <= j+j";
by (fast_arith_tac 1);
Goal "!!a::int. [| a+b < i+j; a<b; i<j |] \
\     ==> a+a - - -1 < j+j - 3";
by (fast_arith_tac 1);
Goal "!!a::int. a+b+c <= i+j+k & a<=b & b<=c & i<=j & j<=k --> a+a+a <= k+k+k";
by (arith_tac 1);
Goal "!!a::int. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a <= l";
by (fast_arith_tac 1);
Goal "!!a::int. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a+a+a+a <= l+l+l+l";
by (fast_arith_tac 1);
Goal "!!a::int. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a+a+a+a+a <= l+l+l+l+i";
by (fast_arith_tac 1);
Goal "!!a::int. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a+a+a+a+a+a <= l+l+l+l+i+l";
by (fast_arith_tac 1);
Goal "!!a::int. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> 6*a <= 5*l+i";
by (fast_arith_tac 1);
*)
