(*  Title:      Pure/Proof/extraction.ML
    ID:         $Id$
    Author:     Stefan Berghofer, TU Muenchen

Extraction of programs from proofs.
*)

signature EXTRACTION =
sig
  val set_preprocessor : (Sign.sg -> Proofterm.proof -> Proofterm.proof) -> theory -> theory
  val add_realizes_eqns_i : ((term * term) list * (term * term)) list -> theory -> theory
  val add_realizes_eqns : string list -> theory -> theory
  val add_typeof_eqns_i : ((term * term) list * (term * term)) list -> theory -> theory
  val add_typeof_eqns : string list -> theory -> theory
  val add_realizers_i : (string * (string list * term * Proofterm.proof)) list
    -> theory -> theory
  val add_realizers : (thm * (string list * string * string)) list
    -> theory -> theory
  val add_expand_thms : thm list -> theory -> theory
  val add_types : (xstring * ((term -> term option) list *
    (term -> typ -> term -> typ -> term) option)) list -> theory -> theory
  val extract : (thm * string list) list -> theory -> theory
  val nullT : typ
  val nullt : term
  val mk_typ : typ -> term
  val etype_of : theory -> string list -> typ list -> term -> typ
  val realizes_of: theory -> string list -> term -> term -> term
end;

structure Extraction : EXTRACTION =
struct

open Proofterm;

(**** tools ****)

fun add_syntax thy =
  thy
  |> Theory.copy
  |> Theory.root_path
  |> Theory.add_types [("Type", 0, NoSyn), ("Null", 0, NoSyn)]
  |> Theory.add_consts
      [("typeof", "'b::{} => Type", NoSyn),
       ("Type", "'a::{} itself => Type", NoSyn),
       ("Null", "Null", NoSyn),
       ("realizes", "'a::{} => 'b::{} => 'b", NoSyn)];

val nullT = Type ("Null", []);
val nullt = Const ("Null", nullT);

fun mk_typ T =
  Const ("Type", itselfT T --> Type ("Type", [])) $ Logic.mk_type T;

fun typeof_proc defaultS vs (Const ("typeof", _) $ u) =
      SOME (mk_typ (case strip_comb u of
          (Var ((a, i), _), _) =>
            if a mem vs then TFree ("'" ^ a ^ ":" ^ string_of_int i, defaultS)
            else nullT
        | (Free (a, _), _) =>
            if a mem vs then TFree ("'" ^ a, defaultS) else nullT
        | _ => nullT))
  | typeof_proc _ _ _ = NONE;

fun rlz_proc (Const ("realizes", Type (_, [Type ("Null", []), _])) $ r $ t) = SOME t
  | rlz_proc (Const ("realizes", Type (_, [T, _])) $ r $ t) =
      (case strip_comb t of
         (Var (ixn, U), ts) => SOME (list_comb (Var (ixn, T --> U), r :: ts))
       | (Free (s, U), ts) => SOME (list_comb (Free (s, T --> U), r :: ts))
       | _ => NONE)
  | rlz_proc _ = NONE;

val unpack_ixn = apfst implode o apsnd (fst o read_int o tl) o
  take_prefix (not o equal ":") o explode;

type rules =
  {next: int, rs: ((term * term) list * (term * term)) list,
   net: (int * ((term * term) list * (term * term))) Net.net};

val empty_rules : rules = {next = 0, rs = [], net = Net.empty};

fun add_rule (r as (_, (lhs, _)), {next, rs, net} : rules) =
  {next = next - 1, rs = r :: rs, net = Net.insert_term
     ((Pattern.eta_contract lhs, (next, r)), net, K false)};

fun merge_rules
  ({next, rs = rs1, net} : rules) ({next = next2, rs = rs2, ...} : rules) =
  foldr add_rule {next = next, rs = rs1, net = net} (rs2 \\ rs1);

fun condrew sign rules procs =
  let
    val tsig = Sign.tsig_of sign;

    fun rew tm =
      Pattern.rewrite_term tsig [] (condrew' :: procs) tm
    and condrew' tm =
      let
        val cache = ref ([] : (term * term) list);
        fun lookup f x = (case assoc (!cache, x) of
            NONE =>
              let val y = f x
              in (cache := (x, y) :: !cache; y) end
          | SOME y => y);
      in
        get_first (fn (_, (prems, (tm1, tm2))) =>
        let
          fun ren t = getOpt (Term.rename_abs tm1 tm t, t);
          val inc = Logic.incr_indexes ([], maxidx_of_term tm + 1);
          val env as (Tenv, tenv) = Pattern.match tsig (inc tm1, tm);
          val prems' = map (pairself (Envir.subst_vars env o inc o ren)) prems;
          val env' = Envir.Envir
            {maxidx = Library.foldl Int.max
              (~1, map (Int.max o pairself maxidx_of_term) prems'),
             iTs = Tenv, asol = tenv};
          val env'' = Library.foldl (fn (env, p) =>
            Pattern.unify (sign, env, [pairself (lookup rew) p])) (env', prems')
        in SOME (Envir.norm_term env'' (inc (ren tm2)))
        end handle Pattern.MATCH => NONE | Pattern.Unif => NONE)
          (sort (Int.compare o pairself fst)
            (Net.match_term rules (Pattern.eta_contract tm)))
      end;

  in rew end;

val chtype = change_type o SOME;

fun add_prefix a b = NameSpace.pack (a :: NameSpace.unpack b);

fun corr_name s vs =
  add_prefix "extr" (space_implode "_" (s :: vs)) ^ "_correctness";

fun extr_name s vs = add_prefix "extr" (space_implode "_" (s :: vs));

fun msg d s = priority (implode (replicate d " ") ^ s);

fun vars_of t = rev (foldl_aterms
  (fn (vs, v as Var _) => v ins vs | (vs, _) => vs) ([], t));

fun vfs_of t = vars_of t @ sort (make_ord atless) (term_frees t);

fun forall_intr (t, prop) =
  let val (a, T) = (case t of Var ((a, _), T) => (a, T) | Free p => p)
  in all T $ Abs (a, T, abstract_over (t, prop)) end;

fun forall_intr_prf (t, prf) =
  let val (a, T) = (case t of Var ((a, _), T) => (a, T) | Free p => p)
  in Abst (a, SOME T, prf_abstract_over t prf) end;

val mkabs = foldr (fn (v, t) => Abs ("x", fastype_of v, abstract_over (v, t)));

fun strip_abs 0 t = t
  | strip_abs n (Abs (_, _, t)) = strip_abs (n-1) t
  | strip_abs _ _ = error "strip_abs: not an abstraction";

fun prf_subst_TVars tye =
  map_proof_terms (subst_TVars tye) (typ_subst_TVars tye);

fun relevant_vars types prop = foldr (fn
      (Var ((a, i), T), vs) => (case strip_type T of
        (_, Type (s, _)) => if s mem types then a :: vs else vs
      | _ => vs)
    | (_, vs) => vs) [] (vars_of prop);

fun tname_of (Type (s, _)) = s
  | tname_of _ = "";

fun get_var_type t =
  let
    val vs = Term.add_vars ([], t);
    val fs = Term.add_frees ([], t)
  in fn 
      Var (ixn, _) => (case assoc (Term.add_vars ([], t), ixn) of
          NONE => error "get_var_type: no such variable in term"
        | SOME T => Var (ixn, T))
    | Free (s, _) => (case assoc (Term.add_frees ([], t), s) of
          NONE => error "get_var_type: no such variable in term"
        | SOME T => Free (s, T))
    | _ => error "get_var_type: not a variable"
  end;


(**** theory data ****)

(* data kind 'Pure/extraction' *)

structure ExtractionArgs =
struct
  val name = "Pure/extraction";
  type T =
    {realizes_eqns : rules,
     typeof_eqns : rules,
     types : (string * ((term -> term option) list *
       (term -> typ -> term -> typ -> term) option)) list,
     realizers : (string list * (term * proof)) list Symtab.table,
     defs : thm list,
     expand : (string * term) list,
     prep : (Sign.sg -> proof -> proof) option}

  val empty =
    {realizes_eqns = empty_rules,
     typeof_eqns = empty_rules,
     types = [],
     realizers = Symtab.empty,
     defs = [],
     expand = [],
     prep = NONE};
  val copy = I;
  val prep_ext = I;

  fun merge
    (({realizes_eqns = realizes_eqns1, typeof_eqns = typeof_eqns1, types = types1,
       realizers = realizers1, defs = defs1, expand = expand1, prep = prep1},
      {realizes_eqns = realizes_eqns2, typeof_eqns = typeof_eqns2, types = types2,
       realizers = realizers2, defs = defs2, expand = expand2, prep = prep2}) : T * T) =
    {realizes_eqns = merge_rules realizes_eqns1 realizes_eqns2,
     typeof_eqns = merge_rules typeof_eqns1 typeof_eqns2,
     types = merge_alists types1 types2,
     realizers = Symtab.merge_multi' (eq_set o pairself #1)
       (realizers1, realizers2),
     defs = gen_merge_lists eq_thm defs1 defs2,
     expand = merge_lists expand1 expand2,
     prep = (case prep1 of NONE => prep2 | _ => prep1)};

  fun print sg (x : T) = ();
end;

structure ExtractionData = TheoryDataFun(ExtractionArgs);
val _ = Context.add_setup [ExtractionData.init];

fun read_condeq thy =
  let val sg = sign_of (add_syntax thy)
  in fn s =>
    let val t = Logic.varify (term_of (read_cterm sg (s, propT)))
    in (map Logic.dest_equals (Logic.strip_imp_prems t),
      Logic.dest_equals (Logic.strip_imp_concl t))
    end handle TERM _ => error ("Not a (conditional) meta equality:\n" ^ s)
  end;

(** preprocessor **)

fun set_preprocessor prep thy =
  let val {realizes_eqns, typeof_eqns, types, realizers,
    defs, expand, ...} = ExtractionData.get thy
  in
    ExtractionData.put
      {realizes_eqns = realizes_eqns, typeof_eqns = typeof_eqns, types = types,
       realizers = realizers, defs = defs, expand = expand, prep = SOME prep} thy
  end;

(** equations characterizing realizability **)

fun gen_add_realizes_eqns prep_eq eqns thy =
  let val {realizes_eqns, typeof_eqns, types, realizers,
    defs, expand, prep} = ExtractionData.get thy;
  in
    ExtractionData.put
      {realizes_eqns = foldr add_rule realizes_eqns (map (prep_eq thy) eqns),
       typeof_eqns = typeof_eqns, types = types, realizers = realizers,
       defs = defs, expand = expand, prep = prep} thy
  end

val add_realizes_eqns_i = gen_add_realizes_eqns (K I);
val add_realizes_eqns = gen_add_realizes_eqns read_condeq;

(** equations characterizing type of extracted program **)

fun gen_add_typeof_eqns prep_eq eqns thy =
  let
    val {realizes_eqns, typeof_eqns, types, realizers,
      defs, expand, prep} = ExtractionData.get thy;
    val eqns' = map (prep_eq thy) eqns
  in
    ExtractionData.put
      {realizes_eqns = realizes_eqns, realizers = realizers,
       typeof_eqns = foldr add_rule typeof_eqns eqns',
       types = types, defs = defs, expand = expand, prep = prep} thy
  end

val add_typeof_eqns_i = gen_add_typeof_eqns (K I);
val add_typeof_eqns = gen_add_typeof_eqns read_condeq;

fun thaw (T as TFree (a, S)) =
      if ":" mem explode a then TVar (unpack_ixn a, S) else T
  | thaw (Type (a, Ts)) = Type (a, map thaw Ts)
  | thaw T = T;

fun freeze (TVar ((a, i), S)) = TFree (a ^ ":" ^ string_of_int i, S)
  | freeze (Type (a, Ts)) = Type (a, map freeze Ts)
  | freeze T = T;

fun freeze_thaw f x =
  map_term_types thaw (f (map_term_types freeze x));

fun etype_of sg vs Ts t =
  let
    val {typeof_eqns, ...} = ExtractionData.get_sg sg;
    fun err () = error ("Unable to determine type of extracted program for\n" ^
      Sign.string_of_term sg t)
  in case strip_abs_body (freeze_thaw (condrew sg (#net typeof_eqns)
    [typeof_proc (Sign.defaultS sg) vs]) (list_abs (map (pair "x") (rev Ts),
      Const ("typeof", fastype_of1 (Ts, t) --> Type ("Type", [])) $ t))) of
      Const ("Type", _) $ u => (Logic.dest_type u handle TERM _ => err ())
    | _ => err ()
  end;

(** realizers for axioms / theorems, together with correctness proofs **)

fun gen_add_realizers prep_rlz rs thy =
  let val {realizes_eqns, typeof_eqns, types, realizers,
    defs, expand, prep} = ExtractionData.get thy
  in
    ExtractionData.put
      {realizes_eqns = realizes_eqns, typeof_eqns = typeof_eqns, types = types,
       realizers = foldr Symtab.update_multi
         realizers (map (prep_rlz thy) (rev rs)),
       defs = defs, expand = expand, prep = prep} thy
  end

fun prep_realizer thy =
  let
    val {realizes_eqns, typeof_eqns, defs, types, ...} =
      ExtractionData.get thy;
    val procs = List.concat (map (fst o snd) types);
    val rtypes = map fst types;
    val eqns = Net.merge (#net realizes_eqns, #net typeof_eqns, K false);
    val thy' = add_syntax thy;
    val sign = sign_of thy';
    val tsg = Sign.tsig_of sign;
    val rd = ProofSyntax.read_proof thy' false
  in fn (thm, (vs, s1, s2)) =>
    let
      val name = Thm.name_of_thm thm;
      val _ = assert (name <> "") "add_realizers: unnamed theorem";
      val prop = Pattern.rewrite_term tsg
        (map (Logic.dest_equals o prop_of) defs) [] (prop_of thm);
      val vars = vars_of prop;
      val vars' = filter_out (fn v =>
        tname_of (body_type (fastype_of v)) mem rtypes) vars;
      val T = etype_of sign vs [] prop;
      val (T', thw) = Type.freeze_thaw_type
        (if T = nullT then nullT else map fastype_of vars' ---> T);
      val t = map_term_types thw (term_of (read_cterm sign (s1, T')));
      val r' = freeze_thaw (condrew sign eqns
        (procs @ [typeof_proc (Sign.defaultS sign) vs, rlz_proc]))
          (Const ("realizes", T --> propT --> propT) $
            (if T = nullT then t else list_comb (t, vars')) $ prop);
      val r = foldr forall_intr r' (map (get_var_type r') vars);
      val prf = Reconstruct.reconstruct_proof sign r (rd s2);
    in (name, (vs, (t, prf))) end
  end;

val add_realizers_i = gen_add_realizers
  (fn _ => fn (name, (vs, t, prf)) => (name, (vs, (t, prf))));
val add_realizers = gen_add_realizers prep_realizer;

fun realizes_of thy vs t prop =
  let
    val thy' = add_syntax thy;
    val sign = sign_of thy';
    val {realizes_eqns, typeof_eqns, defs, types, ...} =
      ExtractionData.get thy';
    val procs = List.concat (map (fst o snd) types);
    val eqns = Net.merge (#net realizes_eqns, #net typeof_eqns, K false);
    val prop' = Pattern.rewrite_term (Sign.tsig_of sign)
      (map (Logic.dest_equals o prop_of) defs) [] prop;
  in freeze_thaw (condrew sign eqns
    (procs @ [typeof_proc (Sign.defaultS sign) vs, rlz_proc]))
      (Const ("realizes", fastype_of t --> propT --> propT) $ t $ prop')
  end;

(** expanding theorems / definitions **)

fun add_expand_thm (thy, thm) =
  let
    val {realizes_eqns, typeof_eqns, types, realizers,
      defs, expand, prep} = ExtractionData.get thy;

    val name = Thm.name_of_thm thm;
    val _ = assert (name <> "") "add_expand_thms: unnamed theorem";

    val is_def =
      (case strip_comb (fst (Logic.dest_equals (prop_of thm))) of
         (Const _, ts) => forall is_Var ts andalso null (duplicates ts)
           andalso exists (fn thy =>
               isSome (Symtab.lookup (#axioms (rep_theory thy), name)))
             (thy :: ancestors_of thy)
       | _ => false) handle TERM _ => false;

    val name = Thm.name_of_thm thm;
    val _ = assert (name <> "") "add_expand_thms: unnamed theorem";
  in
    (ExtractionData.put (if is_def then
        {realizes_eqns = realizes_eqns,
         typeof_eqns = add_rule (([],
           Logic.dest_equals (prop_of (Drule.abs_def thm))), typeof_eqns),
         types = types,
         realizers = realizers, defs = gen_ins eq_thm (thm, defs),
         expand = expand, prep = prep}
      else
        {realizes_eqns = realizes_eqns, typeof_eqns = typeof_eqns, types = types,
         realizers = realizers, defs = defs,
         expand = (name, prop_of thm) ins expand, prep = prep}) thy, thm)
  end;

fun add_expand_thms thms thy = Library.foldl (fst o add_expand_thm) (thy, thms);


(** types with computational content **)

fun add_types tys thy =
  let val {realizes_eqns, typeof_eqns, types, realizers,
    defs, expand, prep} = ExtractionData.get thy;
  in
    ExtractionData.put
      {realizes_eqns = realizes_eqns, typeof_eqns = typeof_eqns,
       types = map (apfst (Sign.intern_tycon (sign_of thy))) tys @ types,
       realizers = realizers, defs = defs, expand = expand, prep = prep} thy
  end;


(** Pure setup **)

val _ = Context.add_setup
  [add_types [("prop", ([], NONE))],

   add_typeof_eqns
     ["(typeof (PROP P)) == (Type (TYPE(Null))) ==>  \
    \  (typeof (PROP Q)) == (Type (TYPE('Q))) ==>  \
    \    (typeof (PROP P ==> PROP Q)) == (Type (TYPE('Q)))",

      "(typeof (PROP Q)) == (Type (TYPE(Null))) ==>  \
    \    (typeof (PROP P ==> PROP Q)) == (Type (TYPE(Null)))",

      "(typeof (PROP P)) == (Type (TYPE('P))) ==>  \
    \  (typeof (PROP Q)) == (Type (TYPE('Q))) ==>  \
    \    (typeof (PROP P ==> PROP Q)) == (Type (TYPE('P => 'Q)))",

      "(%x. typeof (PROP P (x))) == (%x. Type (TYPE(Null))) ==>  \
    \    (typeof (!!x. PROP P (x))) == (Type (TYPE(Null)))",

      "(%x. typeof (PROP P (x))) == (%x. Type (TYPE('P))) ==>  \
    \    (typeof (!!x::'a. PROP P (x))) == (Type (TYPE('a => 'P)))",

      "(%x. typeof (f (x))) == (%x. Type (TYPE('f))) ==>  \
    \    (typeof (f)) == (Type (TYPE('f)))"],

   add_realizes_eqns
     ["(typeof (PROP P)) == (Type (TYPE(Null))) ==>  \
    \    (realizes (r) (PROP P ==> PROP Q)) ==  \
    \    (PROP realizes (Null) (PROP P) ==> PROP realizes (r) (PROP Q))",

      "(typeof (PROP P)) == (Type (TYPE('P))) ==>  \
    \  (typeof (PROP Q)) == (Type (TYPE(Null))) ==>  \
    \    (realizes (r) (PROP P ==> PROP Q)) ==  \
    \    (!!x::'P. PROP realizes (x) (PROP P) ==> PROP realizes (Null) (PROP Q))",

      "(realizes (r) (PROP P ==> PROP Q)) ==  \
    \  (!!x. PROP realizes (x) (PROP P) ==> PROP realizes (r (x)) (PROP Q))",

      "(%x. typeof (PROP P (x))) == (%x. Type (TYPE(Null))) ==>  \
    \    (realizes (r) (!!x. PROP P (x))) ==  \
    \    (!!x. PROP realizes (Null) (PROP P (x)))",

      "(realizes (r) (!!x. PROP P (x))) ==  \
    \  (!!x. PROP realizes (r (x)) (PROP P (x)))"],

   Attrib.add_attributes
     [("extraction_expand",
       (Attrib.no_args add_expand_thm, K Attrib.undef_local_attribute),
       "specify theorems / definitions to be expanded during extraction")]];


(**** extract program ****)

val dummyt = Const ("dummy", dummyT);

fun extract thms thy =
  let
    val sg = sign_of (add_syntax thy);
    val tsg = Sign.tsig_of sg;
    val {realizes_eqns, typeof_eqns, types, realizers, defs, expand, prep} =
      ExtractionData.get thy;
    val procs = List.concat (map (fst o snd) types);
    val rtypes = map fst types;
    val typroc = typeof_proc (Sign.defaultS sg);
    val prep = getOpt (prep, K I) sg o ProofRewriteRules.elim_defs sg false defs o
      Reconstruct.expand_proof sg (("", NONE) :: map (apsnd SOME) expand);
    val rrews = Net.merge (#net realizes_eqns, #net typeof_eqns, K false);

    fun find_inst prop Ts ts vs =
      let
        val rvs = relevant_vars rtypes prop;
        val vars = vars_of prop;
        val n = Int.min (length vars, length ts);

        fun add_args ((Var ((a, i), _), t), (vs', tye)) =
          if a mem rvs then
            let val T = etype_of sg vs Ts t
            in if T = nullT then (vs', tye)
               else (a :: vs', (("'" ^ a, i), T) :: tye)
            end
          else (vs', tye)

      in foldr add_args ([], []) (Library.take (n, vars) ~~ Library.take (n, ts)) end;

    fun find vs = Option.map snd o find_first (curry eq_set vs o fst);
    fun find' s = map snd o List.filter (equal s o fst)

    fun app_rlz_rews Ts vs t = strip_abs (length Ts) (freeze_thaw
      (condrew sg rrews (procs @ [typroc vs, rlz_proc])) (list_abs
        (map (pair "x") (rev Ts), t)));

    fun realizes_null vs prop = app_rlz_rews [] vs
      (Const ("realizes", nullT --> propT --> propT) $ nullt $ prop);

    fun corr d defs vs ts Ts hs (PBound i) _ _ = (defs, PBound i)

      | corr d defs vs ts Ts hs (Abst (s, SOME T, prf)) (Abst (_, _, prf')) t =
          let val (defs', corr_prf) = corr d defs vs [] (T :: Ts)
            (dummyt :: hs) prf (incr_pboundvars 1 0 prf')
            (case t of SOME (Abs (_, _, u)) => SOME u | _ => NONE)
          in (defs', Abst (s, SOME T, corr_prf)) end

      | corr d defs vs ts Ts hs (AbsP (s, SOME prop, prf)) (AbsP (_, _, prf')) t =
          let
            val T = etype_of sg vs Ts prop;
            val u = if T = nullT then 
                (case t of SOME u => SOME (incr_boundvars 1 u) | NONE => NONE)
              else (case t of SOME (Abs (_, _, u)) => SOME u | _ => NONE);
            val (defs', corr_prf) = corr d defs vs [] (T :: Ts) (prop :: hs)
              (incr_pboundvars 0 1 prf) (incr_pboundvars 0 1 prf') u;
            val rlz = Const ("realizes", T --> propT --> propT)
          in (defs',
            if T = nullT then AbsP ("R",
              SOME (app_rlz_rews Ts vs (rlz $ nullt $ prop)),
                prf_subst_bounds [nullt] corr_prf)
            else Abst (s, SOME T, AbsP ("R",
              SOME (app_rlz_rews (T :: Ts) vs
                (rlz $ Bound 0 $ incr_boundvars 1 prop)), corr_prf)))
          end

      | corr d defs vs ts Ts hs (prf % SOME t) (prf' % _) t' =
          let
            val (Us, T) = strip_type (fastype_of1 (Ts, t));
            val (defs', corr_prf) = corr d defs vs (t :: ts) Ts hs prf prf'
              (if tname_of T mem rtypes then t'
               else (case t' of SOME (u $ _) => SOME u | _ => NONE));
            val u = if not (tname_of T mem rtypes) then t else
              let
                val eT = etype_of sg vs Ts t;
                val (r, Us') = if eT = nullT then (nullt, Us) else
                  (Bound (length Us), eT :: Us);
                val u = list_comb (incr_boundvars (length Us') t,
                  map Bound (length Us - 1 downto 0));
                val u' = (case assoc (types, tname_of T) of
                    SOME ((_, SOME f)) => f r eT u T
                  | _ => Const ("realizes", eT --> T --> T) $ r $ u)
              in app_rlz_rews Ts vs (list_abs (map (pair "x") Us', u')) end
          in (defs', corr_prf % SOME u) end

      | corr d defs vs ts Ts hs (prf1 %% prf2) (prf1' %% prf2') t =
          let
            val prop = Reconstruct.prop_of' hs prf2';
            val T = etype_of sg vs Ts prop;
            val (defs1, f, u) = if T = nullT then (defs, t, NONE) else
              (case t of
                 SOME (f $ u) => (defs, SOME f, SOME u)
               | _ =>
                 let val (defs1, u) = extr d defs vs [] Ts hs prf2'
                 in (defs1, NONE, SOME u) end)
            val (defs2, corr_prf1) = corr d defs1 vs [] Ts hs prf1 prf1' f;
            val (defs3, corr_prf2) = corr d defs2 vs [] Ts hs prf2 prf2' u;
          in
            if T = nullT then (defs3, corr_prf1 %% corr_prf2) else
              (defs3, corr_prf1 % u %% corr_prf2)
          end

      | corr d defs vs ts Ts hs (prf0 as PThm ((name, _), prf, prop, SOME Ts')) _ _ =
          let
            val (vs', tye) = find_inst prop Ts ts vs;
            val tye' = (map fst (term_tvars prop) ~~ Ts') @ tye;
            val T = etype_of sg vs' [] prop;
            val defs' = if T = nullT then defs
              else fst (extr d defs vs ts Ts hs prf0)
          in
            if T = nullT andalso realizes_null vs' prop aconv prop then (defs, prf0)
            else case Symtab.lookup (realizers, name) of
              NONE => (case find vs' (find' name defs') of
                NONE =>
                  let
                    val _ = assert (T = nullT) "corr: internal error";
                    val _ = msg d ("Building correctness proof for " ^ quote name ^
                      (if null vs' then ""
                       else " (relevant variables: " ^ commas_quote vs' ^ ")"));
                    val prf' = prep (Reconstruct.reconstruct_proof sg prop prf);
                    val (defs'', corr_prf) =
                      corr (d + 1) defs' vs' [] [] [] prf' prf' NONE;
                    val corr_prop = Reconstruct.prop_of corr_prf;
                    val corr_prf' = foldr forall_intr_prf
                      (proof_combt
                         (PThm ((corr_name name vs', []), corr_prf, corr_prop,
                             SOME (map TVar (term_tvars corr_prop))), vfs_of corr_prop))
		      (map (get_var_type corr_prop) (vfs_of prop))
                  in
                    ((name, (vs', ((nullt, nullt), (corr_prf, corr_prf')))) :: defs'',
                     prf_subst_TVars tye' corr_prf')
                  end
              | SOME (_, (_, prf')) => (defs', prf_subst_TVars tye' prf'))
            | SOME rs => (case find vs' rs of
                SOME (_, prf') => (defs', prf_subst_TVars tye' prf')
              | NONE => error ("corr: no realizer for instance of theorem " ^
                  quote name ^ ":\n" ^ Sign.string_of_term sg (Envir.beta_norm
                    (Reconstruct.prop_of (proof_combt (prf0, ts))))))
          end

      | corr d defs vs ts Ts hs (prf0 as PAxm (s, prop, SOME Ts')) _ _ =
          let
            val (vs', tye) = find_inst prop Ts ts vs;
            val tye' = (map fst (term_tvars prop) ~~ Ts') @ tye
          in
            if etype_of sg vs' [] prop = nullT andalso
              realizes_null vs' prop aconv prop then (defs, prf0)
            else case find vs' (Symtab.lookup_multi (realizers, s)) of
              SOME (_, prf) => (defs, prf_subst_TVars tye' prf)
            | NONE => error ("corr: no realizer for instance of axiom " ^
                quote s ^ ":\n" ^ Sign.string_of_term sg (Envir.beta_norm
                  (Reconstruct.prop_of (proof_combt (prf0, ts)))))
          end

      | corr d defs vs ts Ts hs _ _ _ = error "corr: bad proof"

    and extr d defs vs ts Ts hs (PBound i) = (defs, Bound i)

      | extr d defs vs ts Ts hs (Abst (s, SOME T, prf)) =
          let val (defs', t) = extr d defs vs []
            (T :: Ts) (dummyt :: hs) (incr_pboundvars 1 0 prf)
          in (defs', Abs (s, T, t)) end

      | extr d defs vs ts Ts hs (AbsP (s, SOME t, prf)) =
          let
            val T = etype_of sg vs Ts t;
            val (defs', t) = extr d defs vs [] (T :: Ts) (t :: hs)
              (incr_pboundvars 0 1 prf)
          in (defs',
            if T = nullT then subst_bound (nullt, t) else Abs (s, T, t))
          end

      | extr d defs vs ts Ts hs (prf % SOME t) =
          let val (defs', u) = extr d defs vs (t :: ts) Ts hs prf
          in (defs',
            if tname_of (body_type (fastype_of1 (Ts, t))) mem rtypes then u
            else u $ t)
          end

      | extr d defs vs ts Ts hs (prf1 %% prf2) =
          let
            val (defs', f) = extr d defs vs [] Ts hs prf1;
            val prop = Reconstruct.prop_of' hs prf2;
            val T = etype_of sg vs Ts prop
          in
            if T = nullT then (defs', f) else
              let val (defs'', t) = extr d defs' vs [] Ts hs prf2
              in (defs'', f $ t) end
          end

      | extr d defs vs ts Ts hs (prf0 as PThm ((s, _), prf, prop, SOME Ts')) =
          let
            val (vs', tye) = find_inst prop Ts ts vs;
            val tye' = (map fst (term_tvars prop) ~~ Ts') @ tye
          in
            case Symtab.lookup (realizers, s) of
              NONE => (case find vs' (find' s defs) of
                NONE =>
                  let
                    val _ = msg d ("Extracting " ^ quote s ^
                      (if null vs' then ""
                       else " (relevant variables: " ^ commas_quote vs' ^ ")"));
                    val prf' = prep (Reconstruct.reconstruct_proof sg prop prf);
                    val (defs', t) = extr (d + 1) defs vs' [] [] [] prf';
                    val (defs'', corr_prf) =
                      corr (d + 1) defs' vs' [] [] [] prf' prf' (SOME t);

                    val nt = Envir.beta_norm t;
                    val args = filter_out (fn v => tname_of (body_type
                      (fastype_of v)) mem rtypes) (vfs_of prop);
                    val args' = List.filter (fn v => Logic.occs (v, nt)) args;
                    val t' = mkabs nt args';
                    val T = fastype_of t';
                    val cname = extr_name s vs';
                    val c = Const (cname, T);
                    val u = mkabs (list_comb (c, args')) args;
                    val eqn = Logic.mk_equals (c, t');
                    val rlz =
                      Const ("realizes", fastype_of nt --> propT --> propT);
                    val lhs = app_rlz_rews [] vs' (rlz $ nt $ prop);
                    val rhs = app_rlz_rews [] vs' (rlz $ list_comb (c, args') $ prop);
                    val f = app_rlz_rews [] vs'
                      (Abs ("x", T, rlz $ list_comb (Bound 0, args') $ prop));

                    val corr_prf' =
                      chtype [] equal_elim_axm %> lhs %> rhs %%
                       (chtype [propT] symmetric_axm %> rhs %> lhs %%
                         (chtype [propT, T] combination_axm %> f %> f %> c %> t' %%
                           (chtype [T --> propT] reflexive_axm %> f) %%
                           PAxm (cname ^ "_def", eqn,
                             SOME (map TVar (term_tvars eqn))))) %% corr_prf;
                    val corr_prop = Reconstruct.prop_of corr_prf';
                    val corr_prf'' = foldr forall_intr_prf
                      (proof_combt
                        (PThm ((corr_name s vs', []), corr_prf', corr_prop,
                          SOME (map TVar (term_tvars corr_prop))),  vfs_of corr_prop))
		      (map (get_var_type corr_prop) (vfs_of prop));
                  in
                    ((s, (vs', ((t', u), (corr_prf', corr_prf'')))) :: defs'',
                     subst_TVars tye' u)
                  end
              | SOME ((_, u), _) => (defs, subst_TVars tye' u))
            | SOME rs => (case find vs' rs of
                SOME (t, _) => (defs, subst_TVars tye' t)
              | NONE => error ("extr: no realizer for instance of theorem " ^
                  quote s ^ ":\n" ^ Sign.string_of_term sg (Envir.beta_norm
                    (Reconstruct.prop_of (proof_combt (prf0, ts))))))
          end

      | extr d defs vs ts Ts hs (prf0 as PAxm (s, prop, SOME Ts')) =
          let
            val (vs', tye) = find_inst prop Ts ts vs;
            val tye' = (map fst (term_tvars prop) ~~ Ts') @ tye
          in
            case find vs' (Symtab.lookup_multi (realizers, s)) of
              SOME (t, _) => (defs, subst_TVars tye' t)
            | NONE => error ("extr: no realizer for instance of axiom " ^
                quote s ^ ":\n" ^ Sign.string_of_term sg (Envir.beta_norm
                  (Reconstruct.prop_of (proof_combt (prf0, ts)))))
          end

      | extr d defs vs ts Ts hs _ = error "extr: bad proof";

    fun prep_thm (thm, vs) =
      let
        val {prop, der = (_, prf), sign, ...} = rep_thm thm;
        val name = Thm.name_of_thm thm;
        val _ = assert (name <> "") "extraction: unnamed theorem";
        val _ = assert (etype_of sg vs [] prop <> nullT) ("theorem " ^
          quote name ^ " has no computational content")
      in (Reconstruct.reconstruct_proof sign prop prf, vs) end;

    val defs = Library.foldl (fn (defs, (prf, vs)) =>
      fst (extr 0 defs vs [] [] [] prf)) ([], map prep_thm thms);
    val {path, ...} = Sign.rep_sg sg;

    fun add_def ((s, (vs, ((t, u), (prf, _)))), thy) =
      (case Sign.const_type (sign_of thy) (extr_name s vs) of
         NONE =>
           let
             val corr_prop = Reconstruct.prop_of prf;
             val ft = fst (Type.freeze_thaw t);
             val fu = fst (Type.freeze_thaw u);
             val thy' = if t = nullt then thy else thy |>
               Theory.add_consts_i [(extr_name s vs, fastype_of ft, NoSyn)] |>
               fst o PureThy.add_defs_i false [((extr_name s vs ^ "_def",
                 Logic.mk_equals (head_of (strip_abs_body fu), ft)), [])];
           in
             fst (PureThy.store_thm ((corr_name s vs,
               Thm.varifyT (funpow (length (term_vars corr_prop))
                 (forall_elim_var 0) (forall_intr_frees
                   (ProofChecker.thm_of_proof thy'
                     (fst (Proofterm.freeze_thaw_prf prf)))))), []) thy')
           end
       | SOME _ => thy);

  in thy |>
    Theory.absolute_path |>
    curry (Library.foldr add_def) defs |>
    Theory.add_path (NameSpace.pack (getOpt (path,[])))
  end;


(**** interface ****)

structure P = OuterParse and K = OuterSyntax.Keyword;

val parse_vars = Scan.optional (P.$$$ "(" |-- P.list1 P.name --| P.$$$ ")") [];

val realizersP =
  OuterSyntax.command "realizers"
  "specify realizers for primitive axioms / theorems, together with correctness proof"
  K.thy_decl
    (Scan.repeat1 (P.xname -- parse_vars --| P.$$$ ":" -- P.string -- P.string) >>
     (fn xs => Toplevel.theory (fn thy => add_realizers
       (map (fn (((a, vs), s1), s2) =>
         (PureThy.get_thm thy (a, NONE), (vs, s1, s2))) xs) thy)));

val realizabilityP =
  OuterSyntax.command "realizability"
  "add equations characterizing realizability" K.thy_decl
  (Scan.repeat1 P.string >> (Toplevel.theory o add_realizes_eqns));

val typeofP =
  OuterSyntax.command "extract_type"
  "add equations characterizing type of extracted program" K.thy_decl
  (Scan.repeat1 P.string >> (Toplevel.theory o add_typeof_eqns));

val extractP =
  OuterSyntax.command "extract" "extract terms from proofs" K.thy_decl
    (Scan.repeat1 (P.xname -- parse_vars) >> (fn xs => Toplevel.theory
      (fn thy => extract (map (apfst (PureThy.get_thm thy o rpair NONE)) xs) thy)));

val _ = OuterSyntax.add_parsers [realizersP, realizabilityP, typeofP, extractP];

val etype_of = etype_of o sign_of o add_syntax;

end;
