(*  Title:      Pure/Isar/specification.ML
    ID:         $Id$
    Author:     Makarius

Derived local theory specifications --- with type-inference and
toplevel polymorphism.
*)

signature SPECIFICATION =
sig
  val print_consts: local_theory -> (string * typ -> bool) -> (string * typ) list -> unit
  val check_specification: (string * typ option * mixfix) list ->
    ((string * Attrib.src list) * term list) list list -> Proof.context ->
    (((string * typ) * mixfix) list * ((string * Attrib.src list) * term list) list) *
    Proof.context
  val read_specification: (string * string option * mixfix) list ->
    ((string * Attrib.src list) * string list) list list -> Proof.context ->
    (((string * typ) * mixfix) list * ((string * Attrib.src list) * term list) list) *
    Proof.context
  val check_free_specification: (string * typ option * mixfix) list ->
    ((string * Attrib.src list) * term list) list -> Proof.context ->
    (((string * typ) * mixfix) list * ((string * Attrib.src list) * term list) list) *
    Proof.context
  val read_free_specification: (string * string option * mixfix) list ->
    ((string * Attrib.src list) * string list) list -> Proof.context ->
    (((string * typ) * mixfix) list * ((string * Attrib.src list) * term list) list) *
    Proof.context
  val axiomatization: (string * typ option * mixfix) list ->
    ((bstring * Attrib.src list) * term list) list -> local_theory ->
    (term list * (bstring * thm list) list) * local_theory
  val axiomatization_cmd: (string * string option * mixfix) list ->
    ((bstring * Attrib.src list) * string list) list -> local_theory ->
    (term list * (bstring * thm list) list) * local_theory
  val definition:
    (string * typ option * mixfix) option * ((string * Attrib.src list) * term) ->
    local_theory -> (term * (bstring * thm)) * local_theory
  val definition_cmd:
    (string * string option * mixfix) option * ((string * Attrib.src list) * string) ->
    local_theory -> (term * (bstring * thm)) * local_theory
  val abbreviation: Syntax.mode -> (string * typ option * mixfix) option * term ->
    local_theory -> local_theory
  val abbreviation_cmd: Syntax.mode -> (string * string option * mixfix) option * string ->
    local_theory -> local_theory
  val notation: bool -> Syntax.mode -> (term * mixfix) list -> local_theory -> local_theory
  val notation_cmd: bool -> Syntax.mode -> (string * mixfix) list -> local_theory -> local_theory
  val theorems: string -> ((bstring * Attrib.src list) * (thm list * Attrib.src list) list) list ->
    local_theory -> (bstring * thm list) list * local_theory
  val theorems_cmd: string ->
    ((bstring * Attrib.src list) * (Facts.ref * Attrib.src list) list) list ->
    local_theory -> (bstring * thm list) list * local_theory
  val theorem: string -> Method.text option ->
    (thm list list -> local_theory -> local_theory) ->
    string * Attrib.src list -> Element.context_i Locale.element list -> Element.statement_i ->
    bool -> local_theory -> Proof.state
  val theorem_cmd: string -> Method.text option ->
    (thm list list -> local_theory -> local_theory) ->
    string * Attrib.src list -> Element.context Locale.element list -> Element.statement ->
    bool -> local_theory -> Proof.state
  val add_theorem_hook: (bool -> Proof.state -> Proof.state) -> Context.generic -> Context.generic
end;

structure Specification: SPECIFICATION =
struct


(* diagnostics *)

fun print_consts _ _ [] = ()
  | print_consts ctxt pred cs = Pretty.writeln (ProofDisplay.pretty_consts ctxt pred cs);


(* prepare specification *)

local

fun close_forms ctxt i xs As =
  let
    fun add_free (Free (x, _)) = if Variable.is_fixed ctxt x then I else insert (op =) x
      | add_free _ = I;

    val commons = map #1 xs;
    val _ =
      (case duplicates (op =) commons of [] => ()
      | dups => error ("Duplicate local variables " ^ commas_quote dups));
    val frees = rev ((fold o fold_aterms) add_free As (rev commons));
    val types = map (TypeInfer.param i o rpair []) (Name.invents Name.context Name.aT (length frees));
    val uniform_typing = the o AList.lookup (op =) (frees ~~ types);

    fun abs_body lev y (Abs (x, T, b)) = Abs (x, T, abs_body (lev + 1) y b)
      | abs_body lev y (t $ u) = abs_body lev y t $ abs_body lev y u
      | abs_body lev y (t as Free (x, T)) =
          if x = y then TypeInfer.constrain (uniform_typing x) (TypeInfer.constrain T (Bound lev))
          else t
      | abs_body _ _ a = a;
    fun close (y, U) B =
      let val B' = abs_body 0 y (Term.incr_boundvars 1 B)
      in if Term.loose_bvar1 (B', 0) then Term.all dummyT $ Abs (y, U, B') else B end;
    fun close_form A =
      let
        val occ_frees = rev (fold_aterms add_free A []);
        val bounds = xs @ map (rpair dummyT) (subtract (op =) commons occ_frees);
      in fold_rev close bounds A end;
  in map close_form As end;

fun prep_spec prep_vars parse_prop prep_att do_close raw_vars raw_specss ctxt =
  let
    val thy = ProofContext.theory_of ctxt;

    val (vars, vars_ctxt) = ctxt |> prep_vars raw_vars;
    val (xs, params_ctxt) = vars_ctxt |> ProofContext.add_fixes_i vars;

    val Asss = (map o map) (map (parse_prop params_ctxt) o snd) raw_specss;
    val names = Variable.names_of (params_ctxt |> (fold o fold o fold) Variable.declare_term Asss)
      |> fold Name.declare xs;
    val Asss' = #1 ((fold_map o fold_map o fold_map) Term.free_dummy_patterns Asss names);
    val idx = (fold o fold o fold) Term.maxidx_term Asss' ~1 + 1;
    val specs =
      (if do_close then
        #1 (fold_map
            (fn Ass => fn i => (burrow (close_forms params_ctxt i []) Ass, i + 1)) Asss' idx)
      else Asss')
      |> flat |> burrow (Syntax.check_props params_ctxt);
    val specs_ctxt = params_ctxt |> (fold o fold) Variable.declare_term specs;

    val vs = specs_ctxt |> fold_map ProofContext.inferred_param xs |> fst;
    val params = vs ~~ map #3 vars;
    val name_atts = map (fn ((name, atts), _) => (name, map (prep_att thy) atts)) (flat raw_specss);
  in ((params, name_atts ~~ specs), specs_ctxt) end;

fun read_spec x = prep_spec ProofContext.read_vars Syntax.parse_prop Attrib.intern_src x;
fun check_spec x = prep_spec ProofContext.cert_vars (K I) (K I) x;

in

fun read_specification x = read_spec true x;
fun check_specification x = check_spec true x;
fun read_free_specification vars specs = read_spec false vars [specs];
fun check_free_specification vars specs = check_spec false vars [specs];

end;


(* axiomatization *)

fun gen_axioms do_print prep raw_vars raw_specs lthy =
  let
    val ((vars, specs), _) = prep raw_vars [raw_specs] lthy;
    val ((consts, axioms), lthy') = LocalTheory.axioms Thm.axiomK (vars, specs) lthy;
    val consts' = map (Morphism.term (LocalTheory.target_morphism lthy')) consts;
    val _ =
      if not do_print then ()
      else print_consts lthy' (member (op =) (fold Term.add_frees consts' [])) (map fst vars);
  in ((consts, axioms), lthy') end;

val axiomatization = gen_axioms false check_specification;
val axiomatization_cmd = gen_axioms true read_specification;


(* definition *)

fun gen_def do_print prep (raw_var, (raw_a, raw_prop)) lthy =
  let
    val (vars, [((raw_name, atts), [prop])]) =
      fst (prep (the_list raw_var) [(raw_a, [raw_prop])] lthy);
    val (((x, T), rhs), prove) = LocalDefs.derived_def lthy true prop;
    val name = Thm.def_name_optional x raw_name;
    val mx = (case vars of [] => NoSyn | [((x', _), mx)] =>
      if x = x' then mx
      else error ("Head of definition " ^ quote x ^ " differs from declaration " ^ quote x'));
    val ((lhs, (_, th)), lthy2) = lthy
      |> LocalTheory.define Thm.definitionK ((x, mx), ((name ^ "_raw", []), rhs));
    val ((b, [th']), lthy3) = lthy2
      |> LocalTheory.note Thm.definitionK
          ((name, Code.add_default_func_attr :: atts), [prove lthy2 th]);

    val lhs' = Morphism.term (LocalTheory.target_morphism lthy3) lhs;
    val _ =
      if not do_print then ()
      else print_consts lthy3 (member (op =) (Term.add_frees lhs' [])) [(x, T)];
  in ((lhs, (b, th')), lthy3) end;

val definition = gen_def false check_free_specification;
val definition_cmd = gen_def true read_free_specification;


(* abbreviation *)

fun gen_abbrev do_print prep mode (raw_var, raw_prop) lthy =
  let
    val ((vars, [(_, [prop])]), _) =
      prep (the_list raw_var) [(("", []), [raw_prop])]
        (lthy |> ProofContext.set_mode ProofContext.mode_abbrev);
    val ((x, T), rhs) = LocalDefs.abs_def (#2 (LocalDefs.cert_def lthy prop));
    val mx = (case vars of [] => NoSyn | [((y, _), mx)] =>
      if x = y then mx
      else error ("Head of abbreviation " ^ quote x ^ " differs from declaration " ^ quote y));
    val lthy' = lthy
      |> ProofContext.set_syntax_mode mode    (* FIXME ?!? *)
      |> LocalTheory.abbrev mode ((x, mx), rhs) |> snd
      |> ProofContext.restore_syntax_mode lthy;

    val _ = if not do_print then () else print_consts lthy' (K false) [(x, T)];
  in lthy' end;

val abbreviation = gen_abbrev false check_free_specification;
val abbreviation_cmd = gen_abbrev true read_free_specification;


(* notation *)

fun gen_notation prep_const add mode args lthy =
  lthy |> LocalTheory.notation add mode (map (apfst (prep_const lthy)) args);

val notation = gen_notation (K I);
val notation_cmd = gen_notation ProofContext.read_const;


(* fact statements *)

fun gen_theorems prep_fact prep_att kind raw_facts lthy =
  let
    val attrib = prep_att (ProofContext.theory_of lthy);
    val facts = raw_facts |> map (fn ((name, atts), bs) =>
      ((name, map attrib atts),
        bs |> map (fn (b, more_atts) => (prep_fact lthy b, map attrib more_atts))));
    val (res, lthy') = lthy |> LocalTheory.notes kind facts;
    val _ = ProofDisplay.present_results lthy' ((kind, ""), res);
  in (res, lthy') end;

val theorems = gen_theorems (K I) (K I);
val theorems_cmd = gen_theorems ProofContext.get_fact Attrib.intern_src;


(* complex goal statements *)

local

fun subtract_prems ctxt1 ctxt2 =
  Library.drop (length (Assumption.prems_of ctxt1), Assumption.prems_of ctxt2);

fun prep_statement prep_att prep_stmt elems concl ctxt =
  (case concl of
    Element.Shows shows =>
      let
        val (_, loc_ctxt, elems_ctxt, propp) = prep_stmt elems (map snd shows) ctxt;
        val prems = subtract_prems loc_ctxt elems_ctxt;
        val stmt = Attrib.map_specs prep_att (map fst shows ~~ propp);
        val goal_ctxt = fold (fold (Variable.auto_fixes o fst)) propp elems_ctxt;
      in ((prems, stmt, []), goal_ctxt) end
  | Element.Obtains obtains =>
      let
        val case_names = obtains |> map_index
          (fn (i, ("", _)) => string_of_int (i + 1) | (_, (name, _)) => name);
        val constraints = obtains |> map (fn (_, (vars, _)) =>
          Locale.Elem (Element.Constrains
            (vars |> map_filter (fn (x, SOME T) => SOME (x, T) | _ => NONE))));

        val raw_propp = obtains |> map (fn (_, (_, props)) => map (rpair []) props);
        val (_, loc_ctxt, elems_ctxt, propp) = prep_stmt (elems @ constraints) raw_propp ctxt;

        val thesis = ObjectLogic.fixed_judgment (ProofContext.theory_of ctxt) AutoBind.thesisN;

        fun assume_case ((name, (vars, _)), asms) ctxt' =
          let
            val xs = map fst vars;
            val props = map fst asms;
            val (parms, _) = ctxt'
              |> fold Variable.declare_term props
              |> fold_map ProofContext.inferred_param xs;
            val asm = Term.list_all_free (parms, Logic.list_implies (props, thesis));
          in
            ctxt' |> (snd o ProofContext.add_fixes_i (map (fn x => (x, NONE, NoSyn)) xs));
            ctxt' |> Variable.auto_fixes asm
            |> ProofContext.add_assms_i Assumption.assume_export
              [((name, [ContextRules.intro_query NONE]), [(asm, [])])]
            |>> (fn [(_, [th])] => th)
          end;

        val atts = map (Attrib.internal o K)
          [RuleCases.consumes (~ (length obtains)), RuleCases.case_names case_names];
        val prems = subtract_prems loc_ctxt elems_ctxt;
        val stmt = [(("", atts), [(thesis, [])])];

        val (facts, goal_ctxt) = elems_ctxt
          |> (snd o ProofContext.add_fixes_i [(AutoBind.thesisN, NONE, NoSyn)])
          |> fold_map assume_case (obtains ~~ propp)
          |-> (fn ths => ProofContext.note_thmss_i Thm.assumptionK
                [((Obtain.thatN, []), [(ths, [])])] #> #2 #> pair ths);
      in ((prems, stmt, facts), goal_ctxt) end);

structure TheoremHook = GenericDataFun
(
  type T = ((bool -> Proof.state -> Proof.state) * stamp) list;
  val empty = [];
  val extend = I;
  fun merge _ hooks : T = Library.merge (eq_snd op =) hooks;
);

fun gen_theorem prep_att prep_stmt
    kind before_qed after_qed (name, raw_atts) raw_elems raw_concl int lthy0 =
  let
    val _ = LocalTheory.affirm lthy0;
    val thy = ProofContext.theory_of lthy0;

    val (loc, ctxt, lthy) =
      (case (TheoryTarget.peek lthy0, exists (fn Locale.Expr _ => true | _ => false) raw_elems) of
        ({target, is_locale = true, ...}, true) =>
          (*temporary workaround for non-modularity of in/includes*)  (* FIXME *)
          (SOME target, ProofContext.init thy, LocalTheory.restore lthy0)
      | _ => (NONE, lthy0, lthy0));

    val attrib = prep_att thy;
    val atts = map attrib raw_atts;

    val elems = raw_elems |> map (Locale.map_elem (Element.map_ctxt_attrib attrib));
    val ((prems, stmt, facts), goal_ctxt) =
      prep_statement attrib (prep_stmt loc) elems raw_concl ctxt;

    fun after_qed' results goal_ctxt' =
      let val results' =
        burrow (map Goal.norm_result o ProofContext.export goal_ctxt' lthy) results
      in
        lthy
        |> LocalTheory.notes kind (map2 (fn (a, _) => fn ths => (a, [(ths, [])])) stmt results')
        |> (fn (res, lthy') =>
          (ProofDisplay.present_results lthy' ((kind, name), res);
            if name = "" andalso null atts then lthy'
            else #2 (LocalTheory.notes kind [((name, atts), [(maps #2 res, [])])] lthy')))
        |> after_qed results'
      end;
  in
    goal_ctxt
    |> ProofContext.note_thmss_i Thm.assumptionK [((AutoBind.assmsN, []), [(prems, [])])]
    |> snd
    |> Proof.theorem_i before_qed after_qed' (map snd stmt)
    |> Proof.refine_insert facts
    |> Library.apply (map (fn (f, _) => f int) (rev (TheoremHook.get (Context.Proof goal_ctxt))))
  end;

in

val theorem = gen_theorem (K I) Locale.cert_context_statement;
val theorem_cmd = gen_theorem Attrib.intern_src Locale.read_context_statement_i;

fun add_theorem_hook f = TheoremHook.map (cons (f, stamp ()));

end;

end;
