(*  Title:      Pure/goal.ML
    ID:         $Id$
    Author:     Makarius and Lawrence C Paulson

Goals in tactical theorem proving.
*)

signature BASIC_GOAL =
sig
  val SELECT_GOAL: tactic -> int -> tactic
end;

signature GOAL =
sig
  include BASIC_GOAL
  val init: cterm -> thm
  val protect: thm -> thm
  val conclude: thm -> thm
  val finish: thm -> thm
  val norm_hhf: thm -> thm
  val norm_hhf_protect: thm -> thm
  val compose_hhf: thm -> int -> thm -> thm Seq.seq
  val compose_hhf_tac: thm -> int -> tactic
  val comp_hhf: thm -> thm -> thm
  val prove_multi: Context.proof -> string list -> term list -> term list ->
    (thm list -> tactic) -> thm list
  val prove: Context.proof -> string list -> term list -> term -> (thm list -> tactic) -> thm
  val prove_global: theory -> string list -> term list -> term -> (thm list -> tactic) -> thm
  val prove_raw: cterm list -> cterm -> (thm list -> tactic) -> thm
  val extract: int -> int -> thm -> thm Seq.seq
  val retrofit: int -> int -> thm -> thm -> thm Seq.seq
end;

structure Goal: GOAL =
struct

(** goals **)

(*
  -------- (init)
  C ==> #C
*)
fun init ct = Drule.instantiate' [] [SOME ct] Drule.protectI;

(*
   C
  --- (protect)
  #C
*)
fun protect th = th COMP Drule.incr_indexes th Drule.protectI;

(*
  A ==> ... ==> #C
  ---------------- (conclude)
  A ==> ... ==> C
*)
fun conclude th =
  (case SINGLE (Thm.compose_no_flatten false (th, Thm.nprems_of th) 1)
      (Drule.incr_indexes th Drule.protectD) of
    SOME th' => th'
  | NONE => raise THM ("Failed to conclude goal", 0, [th]));

(*
  #C
  --- (finish)
   C
*)
fun finish th =
  (case Thm.nprems_of th of
    0 => conclude th
  | n => raise THM ("Proof failed.\n" ^
      Pretty.string_of (Pretty.chunks (Display.pretty_goals n th)) ^
      ("\n" ^ string_of_int n ^ " unsolved goal(s)!"), 0, [th]));



(** results **)

(* HHF normal form: !! before ==>, outermost !! generalized *)

local

fun gen_norm_hhf ss =
  (not o Drule.is_norm_hhf o Thm.prop_of) ?
    Drule.fconv_rule (MetaSimplifier.rewrite_cterm (true, false, false) (K (K NONE)) ss)
  #> Thm.adjust_maxidx_thm
  #> Drule.gen_all;

val ss =
  MetaSimplifier.theory_context ProtoPure.thy MetaSimplifier.empty_ss
    addsimps [Drule.norm_hhf_eq];

in

val norm_hhf = gen_norm_hhf ss;
val norm_hhf_protect = gen_norm_hhf (ss addeqcongs [Drule.protect_cong]);

end;


(* composition of normal results *)

fun compose_hhf tha i thb =
  Thm.bicompose false (false, Drule.lift_all (Thm.cprem_of thb i) tha, 0) i thb;

fun compose_hhf_tac th i = PRIMSEQ (compose_hhf th i);

fun comp_hhf tha thb =
  (case Seq.chop 2 (compose_hhf tha 1 thb) of
    ([th], _) => th
  | ([], _) => raise THM ("comp_hhf: no unifiers", 1, [tha, thb])
  | _  => raise THM ("comp_hhf: multiple unifiers", 1, [tha, thb]));



(** tactical theorem proving **)

(* prove_multi *)

fun prove_multi ctxt xs asms props tac =
  let
    val thy = Context.theory_of_proof ctxt;
    val string_of_term = Sign.string_of_term thy;

    val prop = Logic.mk_conjunction_list props;
    val statement = Logic.list_implies (asms, prop);

    fun err msg = cat_error msg
      ("The error(s) above occurred for the goal statement:\n" ^ string_of_term statement);
    fun cert_safe t = Thm.cterm_of thy (Envir.beta_norm t)
      handle TERM (msg, _) => err msg | TYPE (msg, _, _) => err msg;

    val _ = cert_safe statement;
    val _ = Term.no_dummy_patterns statement handle TERM (msg, _) => err msg;
    val casms = map cert_safe asms;
    val prems = map (norm_hhf o Thm.assume) casms;

    val ctxt' = ctxt
      |> Variable.set_body false
      |> (snd o Variable.add_fixes xs)
      |> fold Variable.declare_internal (asms @ props);

    val res =
      (case SINGLE (tac prems) (init (cert_safe prop)) of
        NONE => err "Tactic failed."
      | SOME res => res);
    val [results] =
      Conjunction.elim_precise [length props] (finish res) handle THM (msg, _, _) => err msg;
    val _ = Unify.matches_list thy (map (Thm.term_of o cert_safe) props) (map Thm.prop_of results)
      orelse err ("Proved a different theorem: " ^ string_of_term (Thm.prop_of res));
  in
    results
    |> map (Drule.implies_intr_list casms)
    |> Variable.export ctxt' ctxt
    |> map (norm_hhf #> Drule.zero_var_indexes)
  end;


(* prove *)

fun prove ctxt xs asms prop tac = hd (prove_multi ctxt xs asms [prop] tac);

fun prove_global thy xs asms prop tac =
  Drule.standard (prove (Context.init_proof thy) xs asms prop tac);


(* prove_raw -- no checks, no normalization of result! *)

fun prove_raw casms cprop tac =
  (case SINGLE (tac (map (norm_hhf o Thm.assume) casms)) (init cprop) of
    SOME th => Drule.implies_intr_list casms (finish th)
  | NONE => error "Tactic failed.");



(** local goal states **)

fun extract i n st =
  (if i < 1 orelse n < 1 orelse i + n - 1 > Thm.nprems_of st then Seq.empty
   else if n = 1 then Seq.single (Thm.cprem_of st i)
   else Seq.single (foldr1 Conjunction.mk_conjunction (map (Thm.cprem_of st) (i upto i + n - 1))))
  |> Seq.map (Thm.adjust_maxidx #> init);

fun retrofit i n st' st =
  (if n = 1 then st
   else st |> Drule.rotate_prems (i - 1) |> Conjunction.uncurry n |> Drule.rotate_prems (1 - i))
  |> Thm.compose_no_flatten false (conclude st', Thm.nprems_of st') i;

fun SELECT_GOAL tac i st =
  if Thm.nprems_of st = 1 andalso i = 1 then tac st
  else Seq.lifts (retrofit i 1) (Seq.maps tac (extract i 1 st)) st;

end;

structure BasicGoal: BASIC_GOAL = Goal;
open BasicGoal;
