(*  Title:      Pure/Proof/reconstruct.ML
    ID:         $Id$
    Author:     Stefan Berghofer, TU Muenchen

Reconstruction of partial proof terms.
*)

signature RECONSTRUCT =
sig
  val quiet_mode : bool ref
  val reconstruct_proof : theory -> term -> Proofterm.proof -> Proofterm.proof
  val prop_of' : term list -> Proofterm.proof -> term
  val prop_of : Proofterm.proof -> term
  val expand_proof : theory -> (string * term option) list ->
    Proofterm.proof -> Proofterm.proof
end;

structure Reconstruct : RECONSTRUCT =
struct

open Proofterm;

val quiet_mode = ref true;
fun message s = if !quiet_mode then () else writeln s;

fun vars_of t = rev (fold_aterms
  (fn v as Var _ => insert (op =) v | _ => I) t []);

fun forall_intr (t, prop) =
  let val (a, T) = (case t of Var ((a, _), T) => (a, T) | Free p => p)
  in all T $ Abs (a, T, abstract_over (t, prop)) end;

fun forall_intr_vfs prop = foldr forall_intr prop
  (vars_of prop @ sort Term.term_ord (term_frees prop));

fun forall_intr_prf (t, prf) =
  let val (a, T) = (case t of Var ((a, _), T) => (a, T) | Free p => p)
  in Abst (a, SOME T, prf_abstract_over t prf) end;

fun forall_intr_vfs_prf prop prf = foldr forall_intr_prf prf
  (vars_of prop @ sort Term.term_ord (term_frees prop));

fun merge_envs (Envir.Envir {asol=asol1, iTs=iTs1, maxidx=maxidx1})
  (Envir.Envir {asol=asol2, iTs=iTs2, maxidx=maxidx2}) =
    Envir.Envir {asol=Vartab.merge (op =) (asol1, asol2),
                 iTs=Vartab.merge (op =) (iTs1, iTs2),
                 maxidx=Int.max (maxidx1, maxidx2)};


(**** generate constraints for proof term ****)

fun mk_var env Ts T = 
  let val (env', v) = Envir.genvar "a" (env, rev Ts ---> T)
  in (env', list_comb (v, map Bound (length Ts - 1 downto 0))) end;

fun mk_tvar (Envir.Envir {iTs, asol, maxidx}, s) =
  (Envir.Envir {iTs = iTs, asol = asol, maxidx = maxidx+1},
   TVar (("'t", maxidx+1), s));

val mk_abs = fold (fn T => fn u => Abs ("", T, u));

fun unifyT sg env T U =
  let
    val Envir.Envir {asol, iTs, maxidx} = env;
    val (iTs', maxidx') = Sign.typ_unify sg (T, U) (iTs, maxidx)
  in Envir.Envir {asol=asol, iTs=iTs', maxidx=maxidx'} end
  handle Type.TUNIFY => error ("Non-unifiable types:\n" ^
    Sign.string_of_typ sg T ^ "\n\n" ^ Sign.string_of_typ sg U);

fun chaseT (env as Envir.Envir {iTs, ...}) (T as TVar ixnS) =
      (case Type.lookup (iTs, ixnS) of NONE => T | SOME T' => chaseT env T')
  | chaseT _ T = T;

fun infer_type sg (env as Envir.Envir {maxidx, asol, iTs}) Ts vTs
      (t as Const (s, T)) = if T = dummyT then (case Sign.const_type sg s of
          NONE => error ("reconstruct_proof: No such constant: " ^ quote s)
        | SOME T => 
            let val T' = Type.strip_sorts (Logic.incr_tvar (maxidx + 1) T)
            in (Const (s, T'), T', vTs,
              Envir.Envir {maxidx = maxidx + 1, asol = asol, iTs = iTs})
            end)
      else (t, T, vTs, env)
  | infer_type sg env Ts vTs (t as Free (s, T)) =
      if T = dummyT then (case Symtab.lookup vTs s of
          NONE =>
            let val (env', T) = mk_tvar (env, [])
            in (Free (s, T), T, Symtab.update_new (s, T) vTs, env') end
        | SOME T => (Free (s, T), T, vTs, env))
      else (t, T, vTs, env)
  | infer_type sg env Ts vTs (Var _) = error "reconstruct_proof: internal error"
  | infer_type sg env Ts vTs (Abs (s, T, t)) =
      let
        val (env', T') = if T = dummyT then mk_tvar (env, []) else (env, T);
        val (t', U, vTs', env'') = infer_type sg env' (T' :: Ts) vTs t
      in (Abs (s, T', t'), T' --> U, vTs', env'') end
  | infer_type sg env Ts vTs (t $ u) =
      let
        val (t', T, vTs1, env1) = infer_type sg env Ts vTs t;
        val (u', U, vTs2, env2) = infer_type sg env1 Ts vTs1 u;
      in (case chaseT env2 T of
          Type ("fun", [U', V]) => (t' $ u', V, vTs2, unifyT sg env2 U U')
        | _ =>
          let val (env3, V) = mk_tvar (env2, [])
          in (t' $ u', V, vTs2, unifyT sg env3 T (U --> V)) end)
      end
  | infer_type sg env Ts vTs (t as Bound i) = (t, List.nth (Ts, i), vTs, env);

fun cantunify sg (t, u) = error ("Non-unifiable terms:\n" ^
  Sign.string_of_term sg t ^ "\n\n" ^ Sign.string_of_term sg u);

fun decompose sg Ts (env, p as (t, u)) =
  let fun rigrig (a, T) (b, U) uT ts us = if a <> b then cantunify sg p
    else apsnd flat (foldl_map (decompose sg Ts) (uT env T U, ts ~~ us))
  in case pairself (strip_comb o Envir.head_norm env) p of
      ((Const c, ts), (Const d, us)) => rigrig c d (unifyT sg) ts us
    | ((Free c, ts), (Free d, us)) => rigrig c d (unifyT sg) ts us
    | ((Bound i, ts), (Bound j, us)) =>
        rigrig (i, dummyT) (j, dummyT) (K o K) ts us
    | ((Abs (_, T, t), []), (Abs (_, U, u), [])) =>
        decompose sg (T::Ts) (unifyT sg env T U, (t, u))
    | ((Abs (_, T, t), []), _) =>
        decompose sg (T::Ts) (env, (t, incr_boundvars 1 u $ Bound 0))
    | (_, (Abs (_, T, u), [])) =>
        decompose sg (T::Ts) (env, (incr_boundvars 1 t $ Bound 0, u))
    | _ => (env, [(mk_abs Ts t, mk_abs Ts u)])
  end;

fun make_constraints_cprf sg env cprf =
  let
    fun add_cnstrt Ts prop prf cs env vTs (t, u) =
      let
        val t' = mk_abs Ts t;
        val u' = mk_abs Ts u
      in
        (prop, prf, cs, Pattern.unify sg (t', u') env, vTs)
        handle Pattern.Pattern =>
            let val (env', cs') = decompose sg [] (env, (t', u'))
            in (prop, prf, cs @ cs', env', vTs) end
        | Pattern.Unif =>
            cantunify sg (Envir.norm_term env t', Envir.norm_term env u')
      end;

    fun mk_cnstrts_atom env vTs prop opTs prf =
          let
            val tvars = term_tvars prop;
            val tfrees = term_tfrees prop;
            val (prop', fmap) = Type.varify (prop, []);
            val (env', Ts) = (case opTs of
                NONE => foldl_map mk_tvar (env, map snd tvars @ map snd tfrees)
              | SOME Ts => (env, Ts));
            val prop'' = subst_TVars (map fst tvars @ map snd fmap ~~ Ts)
              (forall_intr_vfs prop') handle Library.UnequalLengths =>
                error ("Wrong number of type arguments for " ^
                  quote (fst (get_name_tags [] prop prf)))
          in (prop'', change_type (SOME Ts) prf, [], env', vTs) end;

    fun head_norm (prop, prf, cnstrts, env, vTs) =
      (Envir.head_norm env prop, prf, cnstrts, env, vTs);
 
    fun mk_cnstrts env _ Hs vTs (PBound i) = (List.nth (Hs, i), PBound i, [], env, vTs)
      | mk_cnstrts env Ts Hs vTs (Abst (s, opT, cprf)) =
          let
            val (env', T) = (case opT of
              NONE => mk_tvar (env, []) | SOME T => (env, T));
            val (t, prf, cnstrts, env'', vTs') =
              mk_cnstrts env' (T::Ts) (map (incr_boundvars 1) Hs) vTs cprf;
          in (Const ("all", (T --> propT) --> propT) $ Abs (s, T, t), Abst (s, SOME T, prf),
            cnstrts, env'', vTs')
          end
      | mk_cnstrts env Ts Hs vTs (AbsP (s, SOME t, cprf)) =
          let
            val (t', _, vTs', env') = infer_type sg env Ts vTs t;
            val (u, prf, cnstrts, env'', vTs'') = mk_cnstrts env' Ts (t'::Hs) vTs' cprf;
          in (Logic.mk_implies (t', u), AbsP (s, SOME t', prf), cnstrts, env'', vTs'')
          end
      | mk_cnstrts env Ts Hs vTs (AbsP (s, NONE, cprf)) =
          let
            val (env', t) = mk_var env Ts propT;
            val (u, prf, cnstrts, env'', vTs') = mk_cnstrts env' Ts (t::Hs) vTs cprf;
          in (Logic.mk_implies (t, u), AbsP (s, SOME t, prf), cnstrts, env'', vTs')
          end
      | mk_cnstrts env Ts Hs vTs (cprf1 %% cprf2) =
          let val (u, prf2, cnstrts, env', vTs') = mk_cnstrts env Ts Hs vTs cprf2
          in (case head_norm (mk_cnstrts env' Ts Hs vTs' cprf1) of
              (Const ("==>", _) $ u' $ t', prf1, cnstrts', env'', vTs'') =>
                add_cnstrt Ts t' (prf1 %% prf2) (cnstrts' @ cnstrts)
                  env'' vTs'' (u, u')
            | (t, prf1, cnstrts', env'', vTs'') =>
                let val (env''', v) = mk_var env'' Ts propT
                in add_cnstrt Ts v (prf1 %% prf2) (cnstrts' @ cnstrts)
                  env''' vTs'' (t, Logic.mk_implies (u, v))
                end)
          end
      | mk_cnstrts env Ts Hs vTs (cprf % SOME t) =
          let val (t', U, vTs1, env1) = infer_type sg env Ts vTs t
          in (case head_norm (mk_cnstrts env1 Ts Hs vTs1 cprf) of
             (Const ("all", Type ("fun", [Type ("fun", [T, _]), _])) $ f,
                 prf, cnstrts, env2, vTs2) =>
               let val env3 = unifyT sg env2 T U
               in (betapply (f, t'), prf % SOME t', cnstrts, env3, vTs2)
               end
           | (u, prf, cnstrts, env2, vTs2) =>
               let val (env3, v) = mk_var env2 Ts (U --> propT);
               in
                 add_cnstrt Ts (v $ t') (prf % SOME t') cnstrts env3 vTs2
                   (u, Const ("all", (U --> propT) --> propT) $ v)
               end)
          end
      | mk_cnstrts env Ts Hs vTs (cprf % NONE) =
          (case head_norm (mk_cnstrts env Ts Hs vTs cprf) of
             (Const ("all", Type ("fun", [Type ("fun", [T, _]), _])) $ f,
                 prf, cnstrts, env', vTs') =>
               let val (env'', t) = mk_var env' Ts T
               in (betapply (f, t), prf % SOME t, cnstrts, env'', vTs')
               end
           | (u, prf, cnstrts, env', vTs') =>
               let
                 val (env1, T) = mk_tvar (env', []);
                 val (env2, v) = mk_var env1 Ts (T --> propT);
                 val (env3, t) = mk_var env2 Ts T
               in
                 add_cnstrt Ts (v $ t) (prf % SOME t) cnstrts env3 vTs'
                   (u, Const ("all", (T --> propT) --> propT) $ v)
               end)
      | mk_cnstrts env _ _ vTs (prf as PThm (_, _, prop, opTs)) =
          mk_cnstrts_atom env vTs prop opTs prf
      | mk_cnstrts env _ _ vTs (prf as PAxm (_, prop, opTs)) =
          mk_cnstrts_atom env vTs prop opTs prf
      | mk_cnstrts env _ _ vTs (prf as Oracle (_, prop, opTs)) =
          mk_cnstrts_atom env vTs prop opTs prf
      | mk_cnstrts env _ _ vTs (Hyp t) = (t, Hyp t, [], env, vTs)
      | mk_cnstrts _ _ _ _ _ = error "reconstruct_proof: minimal proof object"
  in mk_cnstrts env [] [] Symtab.empty cprf end;

fun add_term_ixns (is, t) = fold_aterms (fn Var (xi, _) => insert (op =) xi | _ => I) t is;


(**** update list of free variables of constraints ****)

fun upd_constrs env cs =
  let
    val Envir.Envir {asol, iTs, ...} = env;
    val dom = Vartab.foldl (uncurry (cons o fst) o Library.swap)
      (Vartab.foldl (uncurry (cons o fst) o Library.swap) ([], asol), iTs); 
    val vran = Vartab.foldl (add_typ_ixns o apsnd (snd o snd))
      (Vartab.foldl (add_term_ixns o apsnd (snd o snd)) ([], asol), iTs);
    fun check_cs [] = []
      | check_cs ((u, p, vs)::ps) =
          let val vs' = subtract (op =) dom vs;
          in if vs = vs' then (u, p, vs)::check_cs ps
             else (true, p, vs' union vran)::check_cs ps
          end
  in check_cs cs end;

(**** solution of constraints ****)

fun solve _ [] bigenv = bigenv
  | solve sg cs bigenv =
      let
        fun search env [] = error ("Unsolvable constraints:\n" ^
              Pretty.string_of (Pretty.chunks (map (fn (_, p, _) =>
                Display.pretty_flexpair (Sign.pp sg) (pairself
                  (Envir.norm_term bigenv) p)) cs)))
          | search env ((u, p as (t1, t2), vs)::ps) =
              if u then
                let
                  val tn1 = Envir.norm_term bigenv t1;
                  val tn2 = Envir.norm_term bigenv t2
                in
                  if Pattern.pattern tn1 andalso Pattern.pattern tn2 then
                    (Pattern.unify sg (tn1, tn2) env, ps) handle Pattern.Unif =>
                       cantunify sg (tn1, tn2)
                  else
                    let val (env', cs') = decompose sg [] (env, (tn1, tn2))
                    in if cs' = [(tn1, tn2)] then
                         apsnd (cons (false, (tn1, tn2), vs)) (search env ps)
                       else search env' (map (fn q => (true, q, vs)) cs' @ ps)
                    end
                end
              else apsnd (cons (false, p, vs)) (search env ps);
        val Envir.Envir {maxidx, ...} = bigenv;
        val (env, cs') = search (Envir.empty maxidx) cs;
      in
        solve sg (upd_constrs env cs') (merge_envs bigenv env)
      end;


(**** reconstruction of proofs ****)

fun reconstruct_proof sg prop cprf =
  let
    val (cprf' % SOME prop', thawf) = freeze_thaw_prf (cprf % SOME prop);
    val _ = message "Collecting constraints...";
    val (t, prf, cs, env, _) = make_constraints_cprf sg
      (Envir.empty (maxidx_of_proof cprf)) cprf';
    val cs' = map (fn p => (true, p, op union
      (pairself (map (fst o dest_Var) o term_vars) p))) (map (pairself (Envir.norm_term env)) ((t, prop')::cs));
    val _ = message ("Solving remaining constraints (" ^ string_of_int (length cs') ^ ") ...");
    val env' = solve sg cs' env
  in
    thawf (norm_proof env' prf)
  end;

fun prop_of_atom prop Ts =
  let val (prop', fmap) = Type.varify (prop, []);
  in subst_TVars (map fst (term_tvars prop) @ map snd fmap ~~ Ts)
    (forall_intr_vfs prop')
  end;

val head_norm = Envir.head_norm (Envir.empty 0);

fun prop_of0 Hs (PBound i) = List.nth (Hs, i)
  | prop_of0 Hs (Abst (s, SOME T, prf)) =
      all T $ (Abs (s, T, prop_of0 Hs prf))
  | prop_of0 Hs (AbsP (s, SOME t, prf)) =
      Logic.mk_implies (t, prop_of0 (t :: Hs) prf)
  | prop_of0 Hs (prf % SOME t) = (case head_norm (prop_of0 Hs prf) of
      Const ("all", _) $ f => f $ t
    | _ => error "prop_of: all expected")
  | prop_of0 Hs (prf1 %% prf2) = (case head_norm (prop_of0 Hs prf1) of
      Const ("==>", _) $ P $ Q => Q
    | _ => error "prop_of: ==> expected")
  | prop_of0 Hs (Hyp t) = t
  | prop_of0 Hs (PThm (_, _, prop, SOME Ts)) = prop_of_atom prop Ts
  | prop_of0 Hs (PAxm (_, prop, SOME Ts)) = prop_of_atom prop Ts
  | prop_of0 Hs (Oracle (_, prop, SOME Ts)) = prop_of_atom prop Ts
  | prop_of0 _ _ = error "prop_of: partial proof object";

val prop_of' = Envir.beta_eta_contract oo prop_of0;
val prop_of = prop_of' [];


(**** expand and reconstruct subproofs ****)

fun expand_proof sg thms prf =
  let
    fun expand maxidx prfs (AbsP (s, t, prf)) = 
          let val (maxidx', prfs', prf') = expand maxidx prfs prf
          in (maxidx', prfs', AbsP (s, t, prf')) end
      | expand maxidx prfs (Abst (s, T, prf)) = 
          let val (maxidx', prfs', prf') = expand maxidx prfs prf
          in (maxidx', prfs', Abst (s, T, prf')) end
      | expand maxidx prfs (prf1 %% prf2) =
          let
            val (maxidx', prfs', prf1') = expand maxidx prfs prf1;
            val (maxidx'', prfs'', prf2') = expand maxidx' prfs' prf2;
          in (maxidx'', prfs'', prf1' %% prf2') end
      | expand maxidx prfs (prf % t) =
          let val (maxidx', prfs', prf') = expand maxidx prfs prf
          in (maxidx', prfs', prf' % t) end
      | expand maxidx prfs (prf as PThm ((a, _), cprf, prop, SOME Ts)) =
          if not (exists
            (fn (b, NONE) => a = b
              | (b, SOME prop') => a = b andalso prop = prop') thms)
          then (maxidx, prfs, prf) else
          let
            fun inc i =
              map_proof_terms (Logic.incr_indexes ([], i)) (Logic.incr_tvar i);
            val (maxidx', prf, prfs') =
              (case AList.lookup (op =) prfs (a, prop) of
                NONE =>
                  let
                    val _ = message ("Reconstructing proof of " ^ a);
                    val _ = message (Sign.string_of_term sg prop);
                    val prf' = forall_intr_vfs_prf prop
                      (reconstruct_proof sg prop cprf);
                    val (maxidx', prfs', prf) = expand
                      (maxidx_of_proof prf') prfs prf'
                  in (maxidx' + maxidx + 1, inc (maxidx + 1) prf,
                    ((a, prop), (maxidx', prf)) :: prfs')
                  end
              | SOME (maxidx', prf) => (maxidx' + maxidx + 1,
                  inc (maxidx + 1) prf, prfs));
            val tfrees = term_tfrees prop;
            val tye = map (fn ((s, j), _) => (s, maxidx + 1 + j))
              (term_tvars prop) @ map (rpair ~1 o fst) tfrees ~~ Ts;
            val varify = map_type_tfree (fn p as (a, S) =>
              if p mem tfrees then TVar ((a, ~1), S) else TFree p)
          in
            (maxidx', prfs', map_proof_terms (subst_TVars tye o
               map_term_types varify) (typ_subst_TVars tye o varify) prf)
          end
      | expand maxidx prfs prf = (maxidx, prfs, prf);

  in #3 (expand (maxidx_of_proof prf) [] prf) end;

end;
