(*  Title:      HOL/Hyperreal/hypreal_arith.ML
    ID:         $Id$
    Author:     Tobias Nipkow, TU Muenchen
    Copyright   1999 TU Muenchen

Simprocs for common factor cancellation & Rational coefficient handling

Instantiation of the generic linear arithmetic package for type hypreal.
*)


(****Common factor cancellation****)

local
  open Hyperreal_Numeral_Simprocs
in

val rel_hypreal_number_of = [eq_hypreal_number_of, less_hypreal_number_of,
                          le_hypreal_number_of_eq_not_less];

structure CancelNumeralFactorCommon =
  struct
  val mk_coeff          = mk_coeff
  val dest_coeff        = dest_coeff 1
  val trans_tac         = Real_Numeral_Simprocs.trans_tac
  val norm_tac =
     ALLGOALS (simp_tac (HOL_ss addsimps hypreal_minus_from_mult_simps @ mult_1s))
     THEN ALLGOALS (simp_tac (HOL_ss addsimps bin_simps@hypreal_mult_minus_simps))
     THEN ALLGOALS (simp_tac (HOL_ss addsimps mult_ac))
  val numeral_simp_tac  =
         ALLGOALS (simp_tac (HOL_ss addsimps rel_hypreal_number_of@bin_simps))
  val simplify_meta_eq  = simplify_meta_eq
  end

structure DivCancelNumeralFactor = CancelNumeralFactorFun
 (open CancelNumeralFactorCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_binop "HOL.divide"
  val dest_bal = HOLogic.dest_bin "HOL.divide" hyprealT
  val cancel = mult_divide_cancel_left RS trans
  val neg_exchanges = false
)

structure EqCancelNumeralFactor = CancelNumeralFactorFun
 (open CancelNumeralFactorCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_eq
  val dest_bal = HOLogic.dest_bin "op =" hyprealT
  val cancel = mult_cancel_left RS trans
  val neg_exchanges = false
)

structure LessCancelNumeralFactor = CancelNumeralFactorFun
 (open CancelNumeralFactorCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_binrel "op <"
  val dest_bal = HOLogic.dest_bin "op <" hyprealT
  val cancel = mult_less_cancel_left RS trans
  val neg_exchanges = true
)

structure LeCancelNumeralFactor = CancelNumeralFactorFun
 (open CancelNumeralFactorCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_binrel "op <="
  val dest_bal = HOLogic.dest_bin "op <=" hyprealT
  val cancel = mult_le_cancel_left RS trans
  val neg_exchanges = true
)

val hypreal_cancel_numeral_factors_relations =
  map prep_simproc
   [("hyprealeq_cancel_numeral_factor",
     ["(l::hypreal) * m = n", "(l::hypreal) = m * n"],
     EqCancelNumeralFactor.proc),
    ("hyprealless_cancel_numeral_factor",
     ["(l::hypreal) * m < n", "(l::hypreal) < m * n"],
     LessCancelNumeralFactor.proc),
    ("hyprealle_cancel_numeral_factor",
     ["(l::hypreal) * m <= n", "(l::hypreal) <= m * n"],
     LeCancelNumeralFactor.proc)];

val hypreal_cancel_numeral_factors_divide = prep_simproc
        ("hyprealdiv_cancel_numeral_factor",
         ["((l::hypreal) * m) / n", "(l::hypreal) / (m * n)",
          "((number_of v)::hypreal) / (number_of w)"],
         DivCancelNumeralFactor.proc);

val hypreal_cancel_numeral_factors =
    hypreal_cancel_numeral_factors_relations @
    [hypreal_cancel_numeral_factors_divide];

end;

Addsimprocs hypreal_cancel_numeral_factors;


(*examples:
print_depth 22;
set timing;
set trace_simp;
fun test s = (Goal s; by (Simp_tac 1));

test "0 <= (y::hypreal) * -2";
test "9*x = 12 * (y::hypreal)";
test "(9*x) / (12 * (y::hypreal)) = z";
test "9*x < 12 * (y::hypreal)";
test "9*x <= 12 * (y::hypreal)";

test "-99*x = 123 * (y::hypreal)";
test "(-99*x) / (123 * (y::hypreal)) = z";
test "-99*x < 123 * (y::hypreal)";
test "-99*x <= 123 * (y::hypreal)";

test "999*x = -396 * (y::hypreal)";
test "(999*x) / (-396 * (y::hypreal)) = z";
test "999*x < -396 * (y::hypreal)";
test "999*x <= -396 * (y::hypreal)";

test "-99*x = -81 * (y::hypreal)";
test "(-99*x) / (-81 * (y::hypreal)) = z";
test "-99*x <= -81 * (y::hypreal)";
test "-99*x < -81 * (y::hypreal)";

test "-2 * x = -1 * (y::hypreal)";
test "-2 * x = -(y::hypreal)";
test "(-2 * x) / (-1 * (y::hypreal)) = z";
test "-2 * x < -(y::hypreal)";
test "-2 * x <= -1 * (y::hypreal)";
test "-x < -23 * (y::hypreal)";
test "-x <= -23 * (y::hypreal)";
*)


(** Declarations for ExtractCommonTerm **)

local
  open Hyperreal_Numeral_Simprocs
in

structure CancelFactorCommon =
  struct
  val mk_sum            = long_mk_prod
  val dest_sum          = dest_prod
  val mk_coeff          = mk_coeff
  val dest_coeff        = dest_coeff
  val find_first        = find_first []
  val trans_tac         = Real_Numeral_Simprocs.trans_tac
  val norm_tac = ALLGOALS (simp_tac (HOL_ss addsimps mult_1s@mult_ac))
  end;

structure EqCancelFactor = ExtractCommonTermFun
 (open CancelFactorCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_eq
  val dest_bal = HOLogic.dest_bin "op =" hyprealT
  val simplify_meta_eq  = cancel_simplify_meta_eq mult_cancel_left
);


structure DivideCancelFactor = ExtractCommonTermFun
 (open CancelFactorCommon
  val prove_conv = Bin_Simprocs.prove_conv
  val mk_bal   = HOLogic.mk_binop "HOL.divide"
  val dest_bal = HOLogic.dest_bin "HOL.divide" hyprealT
  val simplify_meta_eq  = cancel_simplify_meta_eq mult_divide_cancel_eq_if
);

val hypreal_cancel_factor =
  map prep_simproc
   [("hypreal_eq_cancel_factor", ["(l::hypreal) * m = n", "(l::hypreal) = m * n"],
     EqCancelFactor.proc),
    ("hypreal_divide_cancel_factor", ["((l::hypreal) * m) / n", "(l::hypreal) / (m * n)"],
     DivideCancelFactor.proc)];

end;

Addsimprocs hypreal_cancel_factor;


(*examples:
print_depth 22;
set timing;
set trace_simp;
fun test s = (Goal s; by (Asm_simp_tac 1));

test "x*k = k*(y::hypreal)";
test "k = k*(y::hypreal)";
test "a*(b*c) = (b::hypreal)";
test "a*(b*c) = d*(b::hypreal)*(x*a)";


test "(x*k) / (k*(y::hypreal)) = (uu::hypreal)";
test "(k) / (k*(y::hypreal)) = (uu::hypreal)";
test "(a*(b*c)) / ((b::hypreal)) = (uu::hypreal)";
test "(a*(b*c)) / (d*(b::hypreal)*(x*a)) = (uu::hypreal)";

(*FIXME: what do we do about this?*)
test "a*(b*c)/(y*z) = d*(b::hypreal)*(x*a)/z";
*)


(****Instantiation of the generic linear arithmetic package****)

val hypreal_mult_left_mono =
    read_instantiate_sg(sign_of HyperBin.thy) [("a","?a::hypreal")] 
    mult_left_mono;


local

(* reduce contradictory <= to False *)
val add_rules =
    [order_less_irrefl, hypreal_numeral_0_eq_0, hypreal_numeral_1_eq_1,
     add_hypreal_number_of, minus_hypreal_number_of, diff_hypreal_number_of,
     mult_hypreal_number_of, eq_hypreal_number_of, less_hypreal_number_of,
     le_hypreal_number_of_eq_not_less, hypreal_diff_def,
     minus_add_distrib, minus_minus, mult_assoc, minus_zero,
     hypreal_add_zero_left, hypreal_add_zero_right,
     hypreal_add_minus, hypreal_add_minus_left,
     mult_zero_left, mult_zero_right,
     mult_1, mult_1_right,
     hypreal_mult_minus_1, hypreal_mult_minus_1_right];

val mono_ss = simpset() addsimps
                [add_mono, add_strict_mono,
                 hypreal_add_less_le_mono,hypreal_add_le_less_mono];

val add_mono_thms_hypreal =
  map (fn s => prove_goal (the_context ()) s
                 (fn prems => [cut_facts_tac prems 1, asm_simp_tac mono_ss 1]))
    ["(i <= j) & (k <= l) ==> i + k <= j + (l::hypreal)",
     "(i  = j) & (k <= l) ==> i + k <= j + (l::hypreal)",
     "(i <= j) & (k  = l) ==> i + k <= j + (l::hypreal)",
     "(i  = j) & (k  = l) ==> i + k  = j + (l::hypreal)",
     "(i < j) & (k = l)   ==> i + k < j + (l::hypreal)",
     "(i = j) & (k < l)   ==> i + k < j + (l::hypreal)",
     "(i < j) & (k <= l)  ==> i + k < j + (l::hypreal)",
     "(i <= j) & (k < l)  ==> i + k < j + (l::hypreal)",
     "(i < j) & (k < l)   ==> i + k < j + (l::hypreal)"];

val simprocs = [Hyperreal_Times_Assoc.conv, 
                Hyperreal_Numeral_Simprocs.combine_numerals,
                hypreal_cancel_numeral_factors_divide]@
               Hyperreal_Numeral_Simprocs.cancel_numerals @
               Hyperreal_Numeral_Simprocs.eval_numerals;

(* reduce contradictory <= to False *)
val simps = [True_implies_equals,
             inst "a" "(number_of ?v)::hypreal" right_distrib,
             divide_1,times_divide_eq_right,times_divide_eq_left];

fun cvar(th,_ $ (_ $ _ $ var)) = cterm_of (#sign(rep_thm th)) var;

val hypreal_mult_mono_thms =
 [(rotate_prems 1 hypreal_mult_less_mono2,
   cvar(hypreal_mult_less_mono2, hd(prems_of hypreal_mult_less_mono2))),
  (hypreal_mult_left_mono,
   cvar(hypreal_mult_left_mono, hd(tl(prems_of hypreal_mult_left_mono))))]

in

val fast_hypreal_arith_simproc =
    Simplifier.simproc (Theory.sign_of (the_context ()))
      "fast_hypreal_arith" 
      ["(m::hypreal) < n", "(m::hypreal) <= n", "(m::hypreal) = n"]
    Fast_Arith.lin_arith_prover;

val hypreal_arith_setup =
 [Fast_Arith.map_data (fn {add_mono_thms, mult_mono_thms, inj_thms, lessD, simpset} =>
   {add_mono_thms = add_mono_thms @ add_mono_thms_hypreal,
    mult_mono_thms = mult_mono_thms @ hypreal_mult_mono_thms,
    inj_thms = inj_thms, (*FIXME: add hypreal*)
    lessD = lessD,  (*Can't change LA_Data_Ref.lessD: the hypreals are dense!*)
    simpset = simpset addsimps add_rules
                      addsimps simps
                      addsimprocs simprocs}),
  arith_discrete ("HyperDef.hypreal",false),
  Simplifier.change_simpset_of (op addsimprocs) [fast_hypreal_arith_simproc]];

end;


(* Some test data [omitting examples that assume the ordering to be discrete!]
Goal "!!a::hypreal. [| a <= b; c <= d; x+y<z |] ==> a+c <= b+d";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a <= b; b+b <= c |] ==> a+a <= c";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a+b <= i+j; a<=b; i<=j |] ==> a+a <= j+j";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. a+b+c <= i+j+k & a<=b & b<=c & i<=j & j<=k --> a+a+a <= k+k+k";
by (arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a <= l";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a+a+a+a <= l+l+l+l";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a+a+a+a+a <= l+l+l+l+i";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> a+a+a+a+a+a <= l+l+l+l+i+l";
by (fast_arith_tac 1);
qed "";

Goal "!!a::hypreal. [| a+b+c+d <= i+j+k+l; a<=b; b<=c; c<=d; i<=j; j<=k; k<=l |] \
\     ==> 6*a <= 5*l+i";
by (fast_arith_tac 1);
qed "";
*)
