(*  Title:      Pure/pure_thy.ML
    ID:         $Id$
    Author:     Markus Wenzel, TU Muenchen

Theorem storage.  The ProtoPure theory.
*)

signature BASIC_PURE_THY =
sig
  datatype interval = FromTo of int * int | From of int | Single of int
  datatype thmref =
    Name of string |
    NameSelection of string * interval list |
    Fact of string
  val get_thm: theory -> thmref -> thm
  val get_thms: theory -> thmref -> thm list
  val get_thmss: theory -> thmref list -> thm list
  structure ProtoPure:
    sig
      val thy: theory
      val prop_def: thm
      val term_def: thm
      val conjunction_def: thm
    end
end;

signature PURE_THY =
sig
  include BASIC_PURE_THY
  val tag_rule: Markup.property -> thm -> thm
  val untag_rule: string -> thm -> thm
  val tag: Markup.property -> attribute
  val untag: string -> attribute
  val has_name_hint: thm -> bool
  val get_name_hint: thm -> string
  val put_name_hint: string -> thm -> thm
  val has_kind: thm -> bool
  val get_kind: thm -> string
  val kind_rule: string -> thm -> thm
  val kind: string -> attribute
  val kind_internal: attribute
  val has_internal: Markup.property list -> bool
  val is_internal: thm -> bool
  val string_of_thmref: thmref -> string
  val get_thm_closure: theory -> thmref -> thm
  val get_thms_closure: theory -> thmref -> thm list
  val single_thm: string -> thm list -> thm
  val name_of_thmref: thmref -> string
  val map_name_of_thmref: (string -> string) -> thmref -> thmref
  val select_thm: thmref -> thm list -> thm list
  val selections: string * thm list -> (thmref * thm) list
  val theorems_of: theory -> thm list NameSpace.table
  val fact_index_of: theory -> FactIndex.T
  val valid_thms: theory -> thmref * thm list -> bool
  val thms_containing: theory -> FactIndex.spec -> (string * thm list) list
  val thms_containing_consts: theory -> string list -> (string * thm) list
  val thms_of: theory -> (string * thm) list
  val all_thms_of: theory -> (string * thm) list
  val hide_thms: bool -> string list -> theory -> theory
  val map_facts: ('a -> 'b) -> ('c * ('a list * 'd) list) list -> ('c * ('b list * 'd) list) list
  val burrow_fact: ('a list -> 'b list) -> ('a list * 'c) list -> ('b list * 'c) list
  val burrow_facts: ('a list -> 'b list) ->
    ('c * ('a list * 'd) list) list -> ('c * ('b list * 'd) list) list
  val name_multi: string -> 'a list -> (string * 'a) list
  val name_thm: bool -> bool -> string -> thm -> thm
  val name_thms: bool -> bool -> string -> thm list -> thm list
  val name_thmss: bool -> string -> (thm list * 'a) list -> (thm list * 'a) list
  val store_thm: (bstring * thm) * attribute list -> theory -> thm * theory
  val smart_store_thms: (bstring * thm list) -> thm list
  val smart_store_thms_open: (bstring * thm list) -> thm list
  val forall_elim_var: int -> thm -> thm
  val forall_elim_vars: int -> thm -> thm
  val add_thms: ((bstring * thm) * attribute list) list -> theory -> thm list * theory
  val add_thmss: ((bstring * thm list) * attribute list) list -> theory -> thm list list * theory
  val note_thmss: string -> ((bstring * attribute list) *
    (thmref * attribute list) list) list -> theory -> (bstring * thm list) list * theory
  val note_thmss_i: string -> ((bstring * attribute list) *
    (thm list * attribute list) list) list -> theory -> (bstring * thm list) list * theory
  val note_thmss_qualified: string -> string -> ((bstring * attribute list) *
    (thm list * attribute list) list) list -> theory -> (bstring * thm list) list * theory
  val add_axioms: ((bstring * string) * attribute list) list -> theory -> thm list * theory
  val add_axioms_i: ((bstring * term) * attribute list) list -> theory -> thm list * theory
  val add_axiomss: ((bstring * string list) * attribute list) list ->
    theory -> thm list list * theory
  val add_axiomss_i: ((bstring * term list) * attribute list) list ->
    theory -> thm list list * theory
  val add_defs: bool -> ((bstring * string) * attribute list) list ->
    theory -> thm list * theory
  val add_defs_i: bool -> ((bstring * term) * attribute list) list ->
    theory -> thm list * theory
  val add_defs_unchecked: bool -> ((bstring * string) * attribute list) list ->
    theory -> thm list * theory
  val add_defs_unchecked_i: bool -> ((bstring * term) * attribute list) list ->
    theory -> thm list * theory
  val appl_syntax: (string * typ * mixfix) list
  val applC_syntax: (string * typ * mixfix) list
end;

structure PureThy: PURE_THY =
struct


(*** theorem tags ***)

(* add / delete tags *)

fun tag_rule tg = Thm.map_tags (insert (op =) tg);
fun untag_rule s = Thm.map_tags (filter_out (fn (s', _) => s = s'));

fun tag tg x = Thm.rule_attribute (K (tag_rule tg)) x;
fun untag s x = Thm.rule_attribute (K (untag_rule s)) x;


(* unofficial theorem names *)

fun the_name_hint thm = the (AList.lookup (op =) (Thm.get_tags thm) Markup.nameN);

val has_name_hint = can the_name_hint;
val get_name_hint = the_default "??.unknown" o try the_name_hint;

fun put_name_hint name = untag_rule Markup.nameN #> tag_rule (Markup.nameN, name);


(* theorem kinds *)

fun the_kind thm = the (AList.lookup (op =) (Thm.get_tags thm) Markup.kindN);

val has_kind = can the_kind;
val get_kind = the_default "??.unknown" o try the_kind;

fun kind_rule k = tag_rule (Markup.kindN, k) o untag_rule Markup.kindN;
fun kind k x = if k = "" then x else Thm.rule_attribute (K (kind_rule k)) x;
fun kind_internal x = kind Thm.internalK x;
fun has_internal tags = exists (fn tg => tg = (Markup.kindN, Thm.internalK)) tags;
val is_internal = has_internal o Thm.get_tags;



(*** theorem database ***)

(** dataype theorems **)

structure TheoremsData = TheoryDataFun
(
  type T =
   {theorems: thm list NameSpace.table,
    index: FactIndex.T} ref;

  fun mk_empty _ =
    ref {theorems = NameSpace.empty_table, index = FactIndex.empty}: T;

  val empty = mk_empty ();
  fun copy (ref x) = ref x;
  val extend = mk_empty;
  fun merge _ = mk_empty;
);

val get_theorems_ref = TheoremsData.get;
val get_theorems = ! o get_theorems_ref;
val theorems_of = #theorems o get_theorems;
val fact_index_of = #index o get_theorems;



(** retrieve theorems **)

fun the_thms _ (SOME thms) = thms
  | the_thms name NONE = error ("Unknown theorem(s) " ^ quote name);

fun single_thm _ [thm] = thm
  | single_thm name _ = error ("Single theorem expected " ^ quote name);


(* datatype interval *)

datatype interval =
  FromTo of int * int |
  From of int |
  Single of int;

fun string_of_interval (FromTo (i, j)) = string_of_int i ^ "-" ^ string_of_int j
  | string_of_interval (From i) = string_of_int i ^ "-"
  | string_of_interval (Single i) = string_of_int i;

fun interval n iv =
  let fun err () = raise Fail ("Bad interval specification " ^ string_of_interval iv) in
    (case iv of
      FromTo (i, j) => if i <= j then i upto j else err ()
    | From i => if i <= n then i upto n else err ()
    | Single i => [i])
  end;


(* datatype thmref *)

datatype thmref =
  Name of string |
  NameSelection of string * interval list |
  Fact of string;

fun name_of_thmref (Name name) = name
  | name_of_thmref (NameSelection (name, _)) = name
  | name_of_thmref (Fact _) = error "Illegal literal fact";

fun map_name_of_thmref f (Name name) = Name (f name)
  | map_name_of_thmref f (NameSelection (name, is)) = NameSelection (f name, is)
  | map_name_of_thmref _ thmref = thmref;

fun string_of_thmref (Name name) = name
  | string_of_thmref (NameSelection (name, is)) =
      name ^ enclose "(" ")" (commas (map string_of_interval is))
  | string_of_thmref (Fact _) = error "Illegal literal fact";


(* select_thm *)

fun select_thm (Name _) thms = thms
  | select_thm (Fact _) thms = thms
  | select_thm (NameSelection (name, ivs)) thms =
      let
        val n = length thms;
        fun err msg = error (msg ^ " for " ^ quote name ^ " (length " ^ string_of_int n ^ ")");
        fun select i =
          if i < 1 orelse i > n then err ("Bad subscript " ^ string_of_int i)
          else nth thms (i - 1);
        val is = maps (interval n) ivs handle Fail msg => err msg;
      in map select is end;


(* selections *)

fun selections (name, [thm]) = [(Name name, thm)]
  | selections (name, thms) = (1 upto length thms, thms) |> ListPair.map (fn (i, thm) =>
      (NameSelection (name, [Single i]), thm));


(* get_thm(s)_closure -- statically scoped versions *)

(*beware of proper order of evaluation!*)

fun lookup_thms thy =
  let
    val (space, thms) = #theorems (get_theorems thy);
    val thy_ref = Theory.check_thy thy;
  in
    fn name =>
      Option.map (map (Thm.transfer (Theory.deref thy_ref)))     (*dynamic identity*)
      (Symtab.lookup thms (NameSpace.intern space name)) (*static content*)
  end;

fun get_thms_closure thy =
  let val closures = map lookup_thms (thy :: Theory.ancestors_of thy) in
    fn thmref =>
      let val name = name_of_thmref thmref;
      in select_thm thmref (the_thms name (get_first (fn f => f name) closures)) end
  end;

fun get_thm_closure thy =
  let val get = get_thms_closure thy
  in fn thmref => single_thm (name_of_thmref thmref) (get thmref) end;


(* get_thms etc. *)

fun get_thms theory thmref =
  let val name = name_of_thmref thmref in
    get_first (fn thy => lookup_thms thy name) (theory :: Theory.ancestors_of theory)
    |> the_thms name |> select_thm thmref |> map (Thm.transfer theory)
  end;

fun get_thmss thy thmrefs = maps (get_thms thy) thmrefs;
fun get_thm thy thmref = single_thm (name_of_thmref thmref) (get_thms thy thmref);


(* thms_containing etc. *)

fun valid_thms thy (thmref, ths) =
  (case try (get_thms thy) thmref of
    NONE => false
  | SOME ths' => Thm.eq_thms (ths, ths'));

fun thms_containing theory spec =
  (theory :: Theory.ancestors_of theory)
  |> maps (fn thy =>
      FactIndex.find (fact_index_of thy) spec
      |> List.filter (fn (name, ths) => valid_thms theory (Name name, ths))
      |> distinct (eq_fst (op =)));

fun thms_containing_consts thy consts =
  thms_containing thy (consts, []) |> maps #2
  |> map (`(get_name_hint));


(* thms_of etc. *)

fun thms_of thy =
  let val thms = #2 (theorems_of thy)
  in map (`(get_name_hint)) (maps snd (Symtab.dest thms)) end;

fun all_thms_of thy = maps thms_of (thy :: Theory.ancestors_of thy);



(** store theorems **)                    (*DESTRUCTIVE*)

(* hiding -- affects current theory node only *)

fun hide_thms fully names thy = CRITICAL (fn () =>
  let
    val r as ref {theorems = (space, thms), index} = get_theorems_ref thy;
    val space' = fold (NameSpace.hide fully) names space;
  in r := {theorems = (space', thms), index = index}; thy end);


(* fact specifications *)

fun map_facts f = map (apsnd (map (apfst (map f))));
fun burrow_fact f = split_list #>> burrow f #> op ~~;
fun burrow_facts f = split_list ##> burrow (burrow_fact f) #> op ~~;


(* naming *)

fun gen_names _ len "" = replicate len ""
  | gen_names j len name = map (fn i => name ^ "_" ^ string_of_int i) (j + 1 upto j + len);

fun name_multi name [x] = [(name, x)]
  | name_multi name xs = gen_names 0 (length xs) name ~~ xs;

fun name_thm pre official name thm = thm
  |> (if Thm.get_name thm <> "" andalso pre orelse not official then I else Thm.put_name name)
  |> (if has_name_hint thm andalso pre orelse name = "" then I else put_name_hint name);

fun name_thms pre official name xs =
  map (uncurry (name_thm pre official)) (name_multi name xs);

fun name_thmss official name fact =
  burrow_fact (name_thms true official name) fact;


(* enter_thms *)

fun warn_overwrite name = warning ("Replaced old copy of theorems " ^ quote name);
fun warn_same name = warning ("Theorem database already contains a copy of " ^ quote name);

fun enter_thms _ _ app_att ("", thms) thy = app_att (thy, thms) |> swap
  | enter_thms pre_name post_name app_att (bname, thms) thy = CRITICAL (fn () =>
      let
        val name = Sign.full_name thy bname;
        val (thy', thms') = apsnd (post_name name) (app_att (thy, pre_name name thms));
        val r as ref {theorems = (space, theorems), index} = get_theorems_ref thy';
        val space' = Sign.declare_name thy' name space;
        val theorems' = Symtab.update (name, thms') theorems;
        val index' = FactIndex.add_global (name, thms') index;
      in
        (case Symtab.lookup theorems name of
          NONE => ()
        | SOME thms'' =>
            if Thm.eq_thms (thms', thms'') then warn_same name
            else warn_overwrite name);
        r := {theorems = (space', theorems'), index = index'};
        (thms', thy')
      end);


(* add_thms(s) *)

fun add_thms_atts pre_name ((bname, thms), atts) =
  enter_thms pre_name (name_thms false true)
    (foldl_map (Thm.theory_attributes atts)) (bname, thms);

fun gen_add_thmss pre_name =
  fold_map (add_thms_atts pre_name);

fun gen_add_thms pre_name args =
  apfst (map hd) o gen_add_thmss pre_name (map (apfst (apsnd single)) args);

val add_thmss = gen_add_thmss (name_thms true true);
val add_thms = gen_add_thms (name_thms true true);


(* note_thmss(_i) *)

local

fun gen_note_thmss get k = fold_map (fn ((bname, more_atts), ths_atts) => fn thy =>
  let
    fun app (x, (ths, atts)) = foldl_map (Thm.theory_attributes atts) (x, ths);
    val (thms, thy') = thy |> enter_thms
      (name_thmss true) (name_thms false true) (apsnd flat o foldl_map app)
      (bname, map (fn (ths, atts) => (get thy ths, atts @ more_atts @ [kind k])) ths_atts);
  in ((bname, thms), thy') end);

in

val note_thmss = gen_note_thmss get_thms;
val note_thmss_i = gen_note_thmss (K I);

end;

fun note_thmss_qualified k path facts thy =
  thy
  |> Sign.add_path path
  |> Sign.no_base_names
  |> note_thmss_i k facts
  ||> Sign.restore_naming thy;


(* store_thm *)

fun store_thm ((bname, thm), atts) thy =
  let val ([th'], thy') = add_thms_atts (name_thms true true) ((bname, [thm]), atts) thy
  in (th', thy') end;


(* smart_store_thms(_open) *)

local

fun smart_store _ (name, []) =
      error ("Cannot store empty list of theorems: " ^ quote name)
  | smart_store official (name, [thm]) =
      fst (enter_thms (name_thms true official) (name_thms false official) I (name, [thm])
        (Thm.theory_of_thm thm))
  | smart_store official (name, thms) =
      let val thy = Theory.merge_list (map Thm.theory_of_thm thms) in
        fst (enter_thms (name_thms true official) (name_thms false official) I (name, thms) thy)
      end;

in

val smart_store_thms = smart_store true;
val smart_store_thms_open = smart_store false;

end;


(* forall_elim_var(s) -- belongs to drule.ML *)

fun forall_elim_vars_aux strip_vars i th =
  let
    val {thy, tpairs, prop, ...} = Thm.rep_thm th;
    val add_used = Term.fold_aterms
      (fn Var ((x, j), _) => if i = j then insert (op =) x else I | _ => I);
    val used = fold (fn (t, u) => add_used t o add_used u) tpairs (add_used prop []);
    val vars = strip_vars prop;
    val cvars = (Name.variant_list used (map #1 vars), vars)
      |> ListPair.map (fn (x, (_, T)) => Thm.cterm_of thy (Var ((x, i), T)));
  in fold Thm.forall_elim cvars th end;

val forall_elim_vars = forall_elim_vars_aux Term.strip_all_vars;

fun forall_elim_var i th = forall_elim_vars_aux
  (fn Const ("all", _) $ Abs (a, T, _) => [(a, T)]
  | _ => raise THM ("forall_elim_vars", i, [th])) i th;


(* store axioms as theorems *)

local
  fun get_ax thy (name, _) = Thm.get_axiom_i thy (Sign.full_name thy name);
  fun get_axs thy named_axs = map (forall_elim_vars 0 o get_ax thy) named_axs;
  fun add_single add ((name, ax), atts) thy =
    let
      val named_ax = [(name, ax)];
      val thy' = add named_ax thy;
      val thm = hd (get_axs thy' named_ax);
    in apfst hd (gen_add_thms (K I) [((name, thm), atts)] thy') end;
  fun add_multi add ((name, axs), atts) thy =
    let
      val named_axs = name_multi name axs;
      val thy' = add named_axs thy;
      val thms = get_axs thy' named_axs;
    in apfst hd (gen_add_thmss (K I) [((name, thms), atts)] thy') end;
  fun add_singles add = fold_map (add_single add);
  fun add_multis add = fold_map (add_multi add);
in
  val add_axioms           = add_singles Theory.add_axioms;
  val add_axioms_i         = add_singles Theory.add_axioms_i;
  val add_axiomss          = add_multis Theory.add_axioms;
  val add_axiomss_i        = add_multis Theory.add_axioms_i;
  val add_defs             = add_singles o Theory.add_defs false;
  val add_defs_i           = add_singles o Theory.add_defs_i false;
  val add_defs_unchecked   = add_singles o Theory.add_defs true;
  val add_defs_unchecked_i = add_singles o Theory.add_defs_i true;
end;



(*** the ProtoPure theory ***)

val typ = SimpleSyntax.read_typ;
val term = SimpleSyntax.read_term;
val prop = SimpleSyntax.read_prop;

val appl_syntax =
 [("_appl", typ "('b => 'a) => args => logic", Mixfix ("(1_/(1'(_')))", [1000, 0], 1000)),
  ("_appl", typ "('b => 'a) => args => aprop", Mixfix ("(1_/(1'(_')))", [1000, 0], 1000))];

val applC_syntax =
 [("",       typ "'a => cargs",                  Delimfix "_"),
  ("_cargs", typ "'a => cargs => cargs",         Mixfix ("_/ _", [1000, 1000], 1000)),
  ("_applC", typ "('b => 'a) => cargs => logic", Mixfix ("(1_/ _)", [1000, 1000], 999)),
  ("_applC", typ "('b => 'a) => cargs => aprop", Mixfix ("(1_/ _)", [1000, 1000], 999))];

val proto_pure =
  Context.pre_pure_thy
  |> Compress.init_data
  |> TheoremsData.init
  |> Sign.add_types
   [("fun", 2, NoSyn),
    ("prop", 0, NoSyn),
    ("itself", 1, NoSyn),
    ("dummy", 0, NoSyn)]
  |> Sign.add_nonterminals Syntax.basic_nonterms
  |> Sign.add_syntax_i
   [("_lambda",     typ "pttrns => 'a => logic",     Mixfix ("(3%_./ _)", [0, 3], 3)),
    ("_abs",        typ "'a",                        NoSyn),
    ("",            typ "'a => args",                Delimfix "_"),
    ("_args",       typ "'a => args => args",        Delimfix "_,/ _"),
    ("",            typ "id => idt",                 Delimfix "_"),
    ("_idtdummy",   typ "idt",                       Delimfix "'_"),
    ("_idtyp",      typ "id => type => idt",         Mixfix ("_::_", [], 0)),
    ("_idtypdummy", typ "type => idt",               Mixfix ("'_()::_", [], 0)),
    ("",            typ "idt => idt",                Delimfix "'(_')"),
    ("",            typ "idt => idts",               Delimfix "_"),
    ("_idts",       typ "idt => idts => idts",       Mixfix ("_/ _", [1, 0], 0)),
    ("",            typ "idt => pttrn",              Delimfix "_"),
    ("",            typ "pttrn => pttrns",           Delimfix "_"),
    ("_pttrns",     typ "pttrn => pttrns => pttrns", Mixfix ("_/ _", [1, 0], 0)),
    ("",            typ "id => aprop",               Delimfix "_"),
    ("",            typ "longid => aprop",           Delimfix "_"),
    ("",            typ "var => aprop",              Delimfix "_"),
    ("_DDDOT",      typ "aprop",                     Delimfix "..."),
    ("_aprop",      typ "aprop => prop",             Delimfix "PROP _"),
    ("_asm",        typ "prop => asms",              Delimfix "_"),
    ("_asms",       typ "prop => asms => asms",      Delimfix "_;/ _"),
    ("_bigimpl",    typ "asms => prop => prop",      Mixfix ("((3[| _ |])/ ==> _)", [0, 1], 1)),
    ("_ofclass",    typ "type => logic => prop",     Delimfix "(1OFCLASS/(1'(_,/ _')))"),
    ("_mk_ofclass", typ "dummy",                     NoSyn),
    ("_TYPE",       typ "type => logic",             Delimfix "(1TYPE/(1'(_')))"),
    ("",            typ "id => logic",               Delimfix "_"),
    ("",            typ "longid => logic",           Delimfix "_"),
    ("",            typ "var => logic",              Delimfix "_"),
    ("_DDDOT",      typ "logic",                     Delimfix "..."),
    ("_constify",   typ "num => num_const",          Delimfix "_"),
    ("_indexnum",   typ "num_const => index",        Delimfix "\\<^sub>_"),
    ("_index",      typ "logic => index",            Delimfix "(00\\<^bsub>_\\<^esub>)"),
    ("_indexdefault", typ "index",                   Delimfix ""),
    ("_indexvar",   typ "index",                     Delimfix "'\\<index>"),
    ("_struct",     typ "index => logic",            Mixfix ("\\<struct>_", [1000], 1000)),
    ("==>",         typ "prop => prop => prop",      Delimfix "op ==>"),
    (Term.dummy_patternN, typ "aprop",               Delimfix "'_")]
  |> Sign.add_syntax_i appl_syntax
  |> Sign.add_modesyntax_i (Symbol.xsymbolsN, true)
   [("fun",      typ "type => type => type",   Mixfix ("(_/ \\<Rightarrow> _)", [1, 0], 0)),
    ("_bracket", typ "types => type => type",  Mixfix ("([_]/ \\<Rightarrow> _)", [0, 0], 0)),
    ("_ofsort",  typ "tid => sort => type",    Mixfix ("_\\<Colon>_", [1000, 0], 1000)),
    ("_constrain", typ "'a => type => 'a",     Mixfix ("_\\<Colon>_", [4, 0], 3)),
    ("_idtyp",    typ "id => type => idt",     Mixfix ("_\\<Colon>_", [], 0)),
    ("_idtypdummy", typ "type => idt",         Mixfix ("'_()\\<Colon>_", [], 0)),
    ("_type_constraint_", typ "'a",            NoSyn),
    ("_lambda",  typ "pttrns => 'a => logic",  Mixfix ("(3\\<lambda>_./ _)", [0, 3], 3)),
    ("==",       typ "'a => 'a => prop",       InfixrName ("\\<equiv>", 2)),
    ("all_binder", typ "idts => prop => prop", Mixfix ("(3\\<And>_./ _)", [0, 0], 0)),
    ("==>",      typ "prop => prop => prop",   InfixrName ("\\<Longrightarrow>", 1)),
    ("_DDDOT",   typ "aprop",                  Delimfix "\\<dots>"),
    ("_bigimpl", typ "asms => prop => prop",   Mixfix ("((1\\<lbrakk>_\\<rbrakk>)/ \\<Longrightarrow> _)", [0, 1], 1)),
    ("_DDDOT",   typ "logic",                  Delimfix "\\<dots>")]
  |> Sign.add_modesyntax_i ("", false)
   [("prop", typ "prop => prop", Mixfix ("_", [0], 0)),
    ("ProtoPure.term", typ "'a => prop", Delimfix "TERM _"),
    ("ProtoPure.conjunction", typ "prop => prop => prop", InfixrName ("&&", 2))]
  |> Sign.add_modesyntax_i ("HTML", false)
   [("_lambda", typ "pttrns => 'a => logic", Mixfix ("(3\\<lambda>_./ _)", [0, 3], 3))]
  |> Sign.add_consts_i
   [("==", typ "'a => 'a => prop", InfixrName ("==", 2)),
    ("==>", typ "prop => prop => prop", Mixfix ("(_/ ==> _)", [2, 1], 1)),
    ("all", typ "('a => prop) => prop", Binder ("!!", 0, 0)),
    ("prop", typ "prop => prop", NoSyn),
    ("TYPE", typ "'a itself", NoSyn),
    (Term.dummy_patternN, typ "'a", Delimfix "'_")]
  |> Theory.add_deps "==" ("==", typ "'a => 'a => prop") []
  |> Theory.add_deps "==>" ("==>", typ "prop => prop => prop") []
  |> Theory.add_deps "all" ("all", typ "('a => prop) => prop") []
  |> Theory.add_deps "TYPE" ("TYPE", typ "'a itself") []
  |> Theory.add_deps Term.dummy_patternN (Term.dummy_patternN, typ "'a") []
  |> Sign.add_trfuns Syntax.pure_trfuns
  |> Sign.add_trfunsT Syntax.pure_trfunsT
  |> Sign.local_path
  |> Sign.add_consts_i
   [("term", typ "'a => prop", NoSyn),
    ("conjunction", typ "prop => prop => prop", NoSyn)]
  |> (add_defs_i false o map Thm.no_attributes)
   [("prop_def", prop "(CONST prop :: prop => prop) (A::prop) == A::prop"),
    ("term_def", prop "(CONST ProtoPure.term :: 'a => prop) (x::'a) == (!!A::prop. A ==> A)"),
    ("conjunction_def", prop "(A && B) == (!!C::prop. (A ==> B ==> C) ==> C)")] |> snd
  |> Sign.hide_consts false ["conjunction", "term"]
  |> add_thmss [(("nothing", []), [])] |> snd
  |> Theory.add_axioms_i Proofterm.equality_axms
  |> Theory.end_theory;

structure ProtoPure =
struct
  val thy = proto_pure;
  val prop_def = get_axiom thy "prop_def";
  val term_def = get_axiom thy "term_def";
  val conjunction_def = get_axiom thy "conjunction_def";
end;

end;

structure BasicPureThy: BASIC_PURE_THY = PureThy;
open BasicPureThy;
