(*  Title:      Pure/codegen.ML
    ID:         $Id$
    Author:     Stefan Berghofer, TU Muenchen

Generic code generator.
*)

signature CODEGEN =
sig
  val quiet_mode : bool ref
  val message : string -> unit
  val mode : string list ref
  val margin : int ref

  datatype 'a mixfix =
      Arg
    | Ignore
    | Module
    | Pretty of Pretty.T
    | Quote of 'a;

  type deftab
  type codegr
  type 'a codegen

  val add_codegen: string -> term codegen -> theory -> theory
  val add_tycodegen: string -> typ codegen -> theory -> theory
  val add_attribute: string -> (Args.T list -> theory attribute * Args.T list) -> theory -> theory
  val add_preprocessor: (theory -> thm list -> thm list) -> theory -> theory
  val preprocess: theory -> thm list -> thm list
  val print_codegens: theory -> unit
  val generate_code: theory -> (string * string) list -> (string * string) list
  val generate_code_i: theory -> (string * term) list -> (string * string) list
  val assoc_consts: (xstring * string option * (term mixfix list *
    (string * string) list)) list -> theory -> theory
  val assoc_consts_i: (xstring * typ option * (term mixfix list *
    (string * string) list)) list -> theory -> theory
  val assoc_types: (xstring * (typ mixfix list *
    (string * string) list)) list -> theory -> theory
  val get_assoc_code: theory -> string -> typ ->
    (term mixfix list * (string * string) list) option
  val get_assoc_type: theory -> string ->
    (typ mixfix list * (string * string) list) option
  val invoke_codegen: theory -> deftab -> string -> string -> bool ->
    codegr * term -> codegr * Pretty.T
  val invoke_tycodegen: theory -> deftab -> string -> string -> bool ->
    codegr * typ -> codegr * Pretty.T
  val mk_id: string -> string
  val mk_const_id: theory -> string -> string -> string -> string
  val mk_type_id: theory -> string -> string -> string -> string
  val thyname_of_type: string -> theory -> string
  val thyname_of_const: string -> theory -> string
  val rename_terms: term list -> term list
  val rename_term: term -> term
  val new_names: term -> string list -> string list
  val new_name: term -> string -> string
  val get_defn: theory -> deftab -> string -> typ ->
    ((typ * (string * (term list * term))) * int option) option
  val is_instance: theory -> typ -> typ -> bool
  val parens: Pretty.T -> Pretty.T
  val mk_app: bool -> Pretty.T -> Pretty.T list -> Pretty.T
  val eta_expand: term -> term list -> int -> term
  val strip_tname: string -> string
  val mk_type: bool -> typ -> Pretty.T
  val mk_term_of: theory -> string -> bool -> typ -> Pretty.T
  val mk_gen: theory -> string -> bool -> string list -> string -> typ -> Pretty.T
  val test_fn: (int -> (string * term) list option) ref
  val test_term: theory -> int -> int -> term -> (string * term) list option
  val parse_mixfix: (string -> 'a) -> string -> 'a mixfix list
  val mk_deftab: theory -> deftab
end;

structure Codegen : CODEGEN =
struct

val quiet_mode = ref true;
fun message s = if !quiet_mode then () else writeln s;

val mode = ref ([] : string list);

val margin = ref 80;

(**** Mixfix syntax ****)

datatype 'a mixfix =
    Arg
  | Ignore
  | Module
  | Pretty of Pretty.T
  | Quote of 'a;

fun is_arg Arg = true
  | is_arg Ignore = true
  | is_arg _ = false;

fun quotes_of [] = []
  | quotes_of (Quote q :: ms) = q :: quotes_of ms
  | quotes_of (_ :: ms) = quotes_of ms;

fun args_of [] xs = ([], xs)
  | args_of (Arg :: ms) (x :: xs) = apfst (cons x) (args_of ms xs)
  | args_of (Ignore :: ms) (_ :: xs) = args_of ms xs
  | args_of (_ :: ms) xs = args_of ms xs;

fun num_args x = length (List.filter is_arg x);


(**** theory data ****)

(* preprocessed definition table *)

type deftab =
  (typ *              (* type of constant *)
    (string *         (* name of theory containing definition of constant *)
      (term list *    (* parameters *)
       term)))        (* right-hand side *)
  list Symtab.table;

(* code dependency graph *)

type codegr =
  (exn option *    (* slot for arbitrary data *)
   string *        (* name of structure containing piece of code *)
   string)         (* piece of code *)
  Graph.T;

(* type of code generators *)

type 'a codegen =
  theory ->    (* theory in which generate_code was called *)
  deftab ->    (* definition table (for efficiency) *)
  codegr ->    (* code dependency graph *)
  string ->    (* node name of caller (for recording dependencies) *)
  string ->    (* theory name of caller (for modular code generation) *)
  bool ->      (* whether to parenthesize generated expression *)
  'a ->        (* item to generate code from *)
  (codegr * Pretty.T) option;

(* parameters for random testing *)

type test_params =
  {size: int, iterations: int, default_type: typ option};

fun merge_test_params
  {size = size1, iterations = iterations1, default_type = default_type1}
  {size = size2, iterations = iterations2, default_type = default_type2} =
  {size = Int.max (size1, size2),
   iterations = Int.max (iterations1, iterations2),
   default_type = case default_type1 of
       NONE => default_type2
     | _ => default_type1};

val default_test_params : test_params =
  {size = 10, iterations = 100, default_type = NONE};

fun set_size size ({iterations, default_type, ...} : test_params) =
  {size = size, iterations = iterations, default_type = default_type};

fun set_iterations iterations ({size, default_type, ...} : test_params) =
  {size = size, iterations = iterations, default_type = default_type};

fun set_default_type s thy ({size, iterations, ...} : test_params) =
  {size = size, iterations = iterations,
   default_type = SOME (typ_of (read_ctyp thy s))};

(* data kind 'Pure/codegen' *)

structure CodegenData = TheoryDataFun
(struct
  val name = "Pure/codegen";
  type T =
    {codegens : (string * term codegen) list,
     tycodegens : (string * typ codegen) list,
     consts : ((string * typ) * (term mixfix list * (string * string) list)) list,
     types : (string * (typ mixfix list * (string * string) list)) list,
     attrs: (string * (Args.T list -> theory attribute * Args.T list)) list,
     preprocs: (stamp * (theory -> thm list -> thm list)) list,
     test_params: test_params};

  val empty =
    {codegens = [], tycodegens = [], consts = [], types = [], attrs = [],
     preprocs = [], test_params = default_test_params};
  val copy = I;
  val extend = I;

  fun merge _
    ({codegens = codegens1, tycodegens = tycodegens1,
      consts = consts1, types = types1, attrs = attrs1,
      preprocs = preprocs1, test_params = test_params1},
     {codegens = codegens2, tycodegens = tycodegens2,
      consts = consts2, types = types2, attrs = attrs2,
      preprocs = preprocs2, test_params = test_params2}) =
    {codegens = merge_alists' codegens1 codegens2,
     tycodegens = merge_alists' tycodegens1 tycodegens2,
     consts = merge_alists consts1 consts2,
     types = merge_alists types1 types2,
     attrs = merge_alists attrs1 attrs2,
     preprocs = merge_alists' preprocs1 preprocs2,
     test_params = merge_test_params test_params1 test_params2};

  fun print _ ({codegens, tycodegens, ...} : T) =
    Pretty.writeln (Pretty.chunks
      [Pretty.strs ("term code generators:" :: map fst codegens),
       Pretty.strs ("type code generators:" :: map fst tycodegens)]);
end);

val _ = Context.add_setup [CodegenData.init];
val print_codegens = CodegenData.print;


(**** access parameters for random testing ****)

fun get_test_params thy = #test_params (CodegenData.get thy);

fun map_test_params f thy =
  let val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
    CodegenData.get thy;
  in CodegenData.put {codegens = codegens, tycodegens = tycodegens,
    consts = consts, types = types, attrs = attrs, preprocs = preprocs,
    test_params = f test_params} thy
  end;


(**** add new code generators to theory ****)

fun add_codegen name f thy =
  let val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
    CodegenData.get thy
  in (case assoc (codegens, name) of
      NONE => CodegenData.put {codegens = (name, f) :: codegens,
        tycodegens = tycodegens, consts = consts, types = types,
        attrs = attrs, preprocs = preprocs, test_params = test_params} thy
    | SOME _ => error ("Code generator " ^ name ^ " already declared"))
  end;

fun add_tycodegen name f thy =
  let val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
    CodegenData.get thy
  in (case assoc (tycodegens, name) of
      NONE => CodegenData.put {tycodegens = (name, f) :: tycodegens,
        codegens = codegens, consts = consts, types = types,
        attrs = attrs, preprocs = preprocs, test_params = test_params} thy
    | SOME _ => error ("Code generator " ^ name ^ " already declared"))
  end;


(**** code attribute ****)

fun add_attribute name att thy =
  let val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
    CodegenData.get thy
  in (case assoc (attrs, name) of
      NONE => CodegenData.put {tycodegens = tycodegens,
        codegens = codegens, consts = consts, types = types,
        attrs = if name = "" then attrs @ [(name, att)] else (name, att) :: attrs,
        preprocs = preprocs,
        test_params = test_params} thy
    | SOME _ => error ("Code attribute " ^ name ^ " already declared"))
  end;

fun mk_parser (a, p) = (if a = "" then Scan.succeed "" else Args.$$$ a) |-- p;

val code_attr =
  Attrib.syntax (Scan.peek (fn thy => foldr op || Scan.fail (map mk_parser
    (#attrs (CodegenData.get thy)))));

val _ = Context.add_setup
 [Attrib.add_attributes
  [("code", (code_attr, K Attrib.undef_local_attribute),
     "declare theorems for code generation")]];


(**** preprocessors ****)

fun add_preprocessor p thy =
  let val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
    CodegenData.get thy
  in CodegenData.put {tycodegens = tycodegens,
    codegens = codegens, consts = consts, types = types,
    attrs = attrs, preprocs = (stamp (), p) :: preprocs,
    test_params = test_params} thy
  end;

fun preprocess thy ths =
  let val {preprocs, ...} = CodegenData.get thy
  in Library.foldl (fn (ths, (_, f)) => f thy ths) (ths, preprocs) end;

fun unfold_attr (thy, eqn) =
  let
    val (name, _) = dest_Const (head_of
      (fst (Logic.dest_equals (prop_of eqn))));
    fun prep thy = map (fn th =>
      if name mem term_consts (prop_of th) then
        rewrite_rule [eqn] (Thm.transfer thy th)
      else th)
  in (add_preprocessor prep thy, eqn) end;

val _ = Context.add_setup [add_attribute "unfold" (Scan.succeed unfold_attr)];


(**** associate constants with target language code ****)

fun gen_assoc_consts prep_type xs thy = Library.foldl (fn (thy, (s, tyopt, syn)) =>
  let
    val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
      CodegenData.get thy;
    val cname = Sign.intern_const thy s;
  in
    (case Sign.const_type thy cname of
       SOME T =>
         let val T' = (case tyopt of
                NONE => T
              | SOME ty =>
                  let val U = prep_type thy ty
                  in if Sign.typ_instance thy (U, T) then U
                    else error ("Illegal type constraint for constant " ^ cname)
                  end)
         in (case assoc (consts, (cname, T')) of
             NONE => CodegenData.put {codegens = codegens,
               tycodegens = tycodegens,
               consts = ((cname, T'), syn) :: consts,
               types = types, attrs = attrs, preprocs = preprocs,
               test_params = test_params} thy
           | SOME _ => error ("Constant " ^ cname ^ " already associated with code"))
         end
     | _ => error ("Not a constant: " ^ s))
  end) (thy, xs);

val assoc_consts_i = gen_assoc_consts (K I);
val assoc_consts = gen_assoc_consts (typ_of oo read_ctyp);


(**** associate types with target language types ****)

fun assoc_types xs thy = Library.foldl (fn (thy, (s, syn)) =>
  let
    val {codegens, tycodegens, consts, types, attrs, preprocs, test_params} =
      CodegenData.get thy;
    val tc = Sign.intern_type thy s
  in
    (case assoc (types, tc) of
       NONE => CodegenData.put {codegens = codegens,
         tycodegens = tycodegens, consts = consts,
         types = (tc, syn) :: types, attrs = attrs,
         preprocs = preprocs, test_params = test_params} thy
     | SOME _ => error ("Type " ^ tc ^ " already associated with code"))
  end) (thy, xs);

fun get_assoc_type thy s = assoc (#types (CodegenData.get thy), s);


(**** make valid ML identifiers ****)

fun is_ascii_letdig x = Symbol.is_ascii_letter x orelse
  Symbol.is_ascii_digit x orelse Symbol.is_ascii_quasi x;

fun dest_sym s = (case split_last (snd (take_prefix (equal "\\") (explode s))) of
    ("<" :: "^" :: xs, ">") => (true, implode xs)
  | ("<" :: xs, ">") => (false, implode xs)
  | _ => sys_error "dest_sym");

fun mk_id s = if s = "" then "" else
  let
    fun check_str [] = []
      | check_str xs = (case take_prefix is_ascii_letdig xs of
          ([], " " :: zs) => check_str zs
        | ([], z :: zs) =>
          if size z = 1 then string_of_int (ord z) :: check_str zs
          else (case dest_sym z of
              (true, "isub") => check_str zs
            | (true, "isup") => "" :: check_str zs
            | (ctrl, s') => (if ctrl then "ctrl_" ^ s' else s') :: check_str zs)
        | (ys, zs) => implode ys :: check_str zs);
    val s' = space_implode "_"
      (List.concat (map (check_str o Symbol.explode) (NameSpace.unpack s)))
  in
    if Symbol.is_ascii_letter (hd (explode s')) then s' else "id_" ^ s'
  end;

fun extrn thy f thyname s =
  let
    val xs = NameSpace.unpack s;
    val s' = setmp NameSpace.long_names false (setmp NameSpace.short_names false
      (setmp NameSpace.unique_names true (f thy))) s;
    val xs' = NameSpace.unpack s'
  in
    if "modular" mem !mode andalso length xs = length xs' andalso hd xs' = thyname
    then NameSpace.pack (tl xs') else s'
  end;

(* thyname:  theory name for caller                                        *)
(* thyname': theory name for callee                                        *)
(* if caller and callee reside in different theories, use qualified access *)

fun mk_const_id thy thyname thyname' s =
  let
    val s' = mk_id (extrn thy Sign.extern_const thyname' s);
    val s'' = if s' mem ThmDatabase.ml_reserved then s' ^ "_const" else s'
  in
    if "modular" mem !mode andalso thyname <> thyname' andalso thyname' <> ""
    then thyname' ^ "." ^ s'' else s''
  end;

fun mk_type_id' f thy thyname thyname' s =
  let
    val s' = mk_id (extrn thy Sign.extern_type thyname' s);
    val s'' = f (if s' mem ThmDatabase.ml_reserved then s' ^ "_type" else s')
  in
    if "modular" mem !mode andalso thyname <> thyname' andalso thyname' <> ""
    then thyname' ^ "." ^ s'' else s''
  end;

val mk_type_id = mk_type_id' I;

fun theory_of_type s thy = 
  if Sign.declared_tyname thy s
  then SOME (if_none (get_first (theory_of_type s) (Theory.parents_of thy)) thy)
  else NONE;

fun theory_of_const s thy = 
  if Sign.declared_const thy s
  then SOME (if_none (get_first (theory_of_const s) (Theory.parents_of thy)) thy)
  else NONE;

fun thyname_of_type s thy = (case theory_of_type s thy of
    NONE => error ("thyname_of_type: no such type: " ^ quote s)
  | SOME thy' => Context.theory_name thy');

fun thyname_of_const s thy = (case theory_of_const s thy of
    NONE => error ("thyname_of_const: no such constant: " ^ quote s)
  | SOME thy' => Context.theory_name thy');

fun rename_terms ts =
  let
    val names = foldr add_term_names
      (map (fst o fst) (Drule.vars_of_terms ts)) ts;
    val reserved = names inter ThmDatabase.ml_reserved;
    val (illegal, alt_names) = split_list (List.mapPartial (fn s =>
      let val s' = mk_id s in if s = s' then NONE else SOME (s, s') end) names)
    val ps = (reserved @ illegal) ~~
      variantlist (map (suffix "'") reserved @ alt_names, names);

    fun rename_id s = getOpt (assoc (ps, s), s);

    fun rename (Var ((a, i), T)) = Var ((rename_id a, i), T)
      | rename (Free (a, T)) = Free (rename_id a, T)
      | rename (Abs (s, T, t)) = Abs (s, T, rename t)
      | rename (t $ u) = rename t $ rename u
      | rename t = t;
  in
    map rename ts
  end;

val rename_term = hd o rename_terms o single;


(**** retrieve definition of constant ****)

fun is_instance thy T1 T2 =
  Sign.typ_instance thy (T1, Type.varifyT T2);

fun get_assoc_code thy s T = Option.map snd (find_first (fn ((s', T'), _) =>
  s = s' andalso is_instance thy T T') (#consts (CodegenData.get thy)));

fun get_aux_code xs = List.mapPartial (fn (m, code) =>
  if m = "" orelse m mem !mode then SOME code else NONE) xs;

fun mk_deftab thy =
  let
    val axmss = map (fn thy' =>
      (Context.theory_name thy', snd (#axioms (Theory.rep_theory thy'))))
      (thy :: Theory.ancestors_of thy);
    fun prep_def def = (case preprocess thy [def] of
      [def'] => prop_of def' | _ => error "mk_deftab: bad preprocessor");
    fun dest t =
      let
        val (lhs, rhs) = Logic.dest_equals t;
        val (c, args) = strip_comb lhs;
        val (s, T) = dest_Const c
      in if forall is_Var args then SOME (s, (T, (args, rhs))) else NONE
      end handle TERM _ => NONE;
    fun add_def thyname (defs, (name, t)) = (case dest t of
        NONE => defs
      | SOME _ => (case dest (prep_def (Thm.get_axiom thy name)) of
          NONE => defs
        | SOME (s, (T, (args, rhs))) => Symtab.update
            ((s, (T, (thyname, split_last (rename_terms (args @ [rhs])))) ::
            if_none (Symtab.lookup (defs, s)) []), defs)))
  in
    foldl (fn ((thyname, axms), defs) =>
      Symtab.foldl (add_def thyname) (defs, axms)) Symtab.empty axmss
  end;

fun get_defn thy defs s T = (case Symtab.lookup (defs, s) of
    NONE => NONE
  | SOME ds =>
      let val i = find_index (is_instance thy T o fst) ds
      in if i >= 0 then
          SOME (List.nth (ds, i), if length ds = 1 then NONE else SOME i)
        else NONE
      end);


(**** invoke suitable code generator for term / type ****)

fun invoke_codegen thy defs dep thyname brack (gr, t) = (case get_first
   (fn (_, f) => f thy defs gr dep thyname brack t) (#codegens (CodegenData.get thy)) of
      NONE => error ("Unable to generate code for term:\n" ^
        Sign.string_of_term thy t ^ "\nrequired by:\n" ^
        commas (Graph.all_succs gr [dep]))
    | SOME x => x);

fun invoke_tycodegen thy defs dep thyname brack (gr, T) = (case get_first
   (fn (_, f) => f thy defs gr dep thyname brack T) (#tycodegens (CodegenData.get thy)) of
      NONE => error ("Unable to generate code for type:\n" ^
        Sign.string_of_typ thy T ^ "\nrequired by:\n" ^
        commas (Graph.all_succs gr [dep]))
    | SOME x => x);


(**** code generator for mixfix expressions ****)

fun parens p = Pretty.block [Pretty.str "(", p, Pretty.str ")"];

fun pretty_fn [] p = [p]
  | pretty_fn (x::xs) p = Pretty.str ("fn " ^ x ^ " =>") ::
      Pretty.brk 1 :: pretty_fn xs p;

fun pretty_mixfix _ _ [] [] _ = []
  | pretty_mixfix module module' (Arg :: ms) (p :: ps) qs =
      p :: pretty_mixfix module module' ms ps qs
  | pretty_mixfix module module' (Ignore :: ms) ps qs =
      pretty_mixfix module module' ms ps qs
  | pretty_mixfix module module' (Module :: ms) ps qs =
      (if "modular" mem !mode andalso module <> module'
       then cons (Pretty.str (module' ^ ".")) else I)
      (pretty_mixfix module module' ms ps qs)
  | pretty_mixfix module module' (Pretty p :: ms) ps qs =
      p :: pretty_mixfix module module' ms ps qs
  | pretty_mixfix module module' (Quote _ :: ms) ps (q :: qs) =
      q :: pretty_mixfix module module' ms ps qs;


(**** default code generators ****)

fun eta_expand t ts i =
  let
    val (Ts, _) = strip_type (fastype_of t);
    val j = i - length ts
  in
    foldr (fn (T, t) => Abs ("x", T, t))
      (list_comb (t, ts @ map Bound (j-1 downto 0))) (Library.take (j, Ts))
  end;

fun mk_app _ p [] = p
  | mk_app brack p ps = if brack then
       Pretty.block (Pretty.str "(" ::
         separate (Pretty.brk 1) (p :: ps) @ [Pretty.str ")"])
     else Pretty.block (separate (Pretty.brk 1) (p :: ps));

fun new_names t xs = variantlist (map mk_id xs,
  map (fst o fst o dest_Var) (term_vars t) union
  add_term_names (t, ThmDatabase.ml_reserved));

fun new_name t x = hd (new_names t [x]);

fun default_codegen thy defs gr dep thyname brack t =
  let
    val (u, ts) = strip_comb t;
    fun codegens brack = foldl_map (invoke_codegen thy defs dep thyname brack)
  in (case u of
      Var ((s, i), T) =>
        let
          val (gr', ps) = codegens true (gr, ts);
          val (gr'', _) = invoke_tycodegen thy defs dep thyname false (gr', T)
        in SOME (gr'', mk_app brack (Pretty.str (s ^
           (if i=0 then "" else string_of_int i))) ps)
        end

    | Free (s, T) =>
        let
          val (gr', ps) = codegens true (gr, ts);
          val (gr'', _) = invoke_tycodegen thy defs dep thyname false (gr', T)
        in SOME (gr'', mk_app brack (Pretty.str s) ps) end

    | Const (s, T) =>
      (case get_assoc_code thy s T of
         SOME (ms, aux) =>
           let val i = num_args ms
           in if length ts < i then
               default_codegen thy defs gr dep thyname brack (eta_expand u ts i)
             else
               let
                 val (ts1, ts2) = args_of ms ts;
                 val (gr1, ps1) = codegens false (gr, ts1);
                 val (gr2, ps2) = codegens true (gr1, ts2);
                 val (gr3, ps3) = codegens false (gr2, quotes_of ms);
                 val (thyname', suffix) = (case get_defn thy defs s T of
                     NONE => (thyname_of_const s thy, "")
                   | SOME ((U, (thyname', _)), NONE) => (thyname', "")
                   | SOME ((U, (thyname', _)), SOME i) =>
                       (thyname', "_def" ^ string_of_int i));
                 val node_id = s ^ suffix;
                 val p = mk_app brack (Pretty.block
                   (pretty_mixfix thyname thyname' ms ps1 ps3)) ps2
               in SOME (case try (Graph.get_node gr3) node_id of
                   NONE => (case get_aux_code aux of
                       [] => (gr3, p)
                     | xs => (Graph.add_edge (node_id, dep) (Graph.new_node
                         (node_id, (NONE, thyname', space_implode "\n" xs ^ "\n")) gr3), p))
                 | SOME _ => (Graph.add_edge (node_id, dep) gr3, p))
               end
           end
       | NONE => (case get_defn thy defs s T of
           NONE => NONE
         | SOME ((U, (thyname', (args, rhs))), k) =>
             let
               val suffix = (case k of NONE => "" | SOME i => "_def" ^ string_of_int i);
               val node_id = s ^ suffix;
               val def_id = mk_const_id thy thyname' thyname' s ^ suffix;
               val call_id = mk_const_id thy thyname thyname' s ^ suffix;
               val (gr', ps) = codegens true (gr, ts);
             in
               SOME (Graph.add_edge (node_id, dep) gr' handle Graph.UNDEF _ =>
                 let
                   val _ = message ("expanding definition of " ^ s);
                   val (Ts, _) = strip_type T;
                   val (args', rhs') =
                     if not (null args) orelse null Ts then (args, rhs) else
                       let val v = Free (new_name rhs "x", hd Ts)
                       in ([v], betapply (rhs, v)) end;
                   val (gr1, p) = invoke_codegen thy defs node_id thyname' false
                     (Graph.add_edge (node_id, dep)
                        (Graph.new_node (node_id, (NONE, "", "")) gr'), rhs');
                   val (gr2, xs) = codegens false (gr1, args');
                   val (gr3, _) = invoke_tycodegen thy defs dep thyname false (gr2, T);
                   val (gr4, ty) = invoke_tycodegen thy defs node_id thyname' false (gr3, U);
                 in Graph.map_node node_id (K (NONE, thyname', Pretty.string_of
                   (Pretty.block (separate (Pretty.brk 1)
                     (if null args' then
                        [Pretty.str ("val " ^ def_id ^ " :"), ty]
                      else Pretty.str ("fun " ^ def_id) :: xs) @
                    [Pretty.str " =", Pretty.brk 1, p, Pretty.str ";"])) ^ "\n\n")) gr4
                 end, mk_app brack (Pretty.str call_id) ps)
             end))

    | Abs _ =>
      let
        val (bs, Ts) = ListPair.unzip (strip_abs_vars u);
        val t = strip_abs_body u
        val bs' = new_names t bs;
        val (gr1, ps) = codegens true (gr, ts);
        val (gr2, p) = invoke_codegen thy defs dep thyname false
          (gr1, subst_bounds (map Free (rev (bs' ~~ Ts)), t));
      in
        SOME (gr2, mk_app brack (Pretty.block (Pretty.str "(" :: pretty_fn bs' p @
          [Pretty.str ")"])) ps)
      end

    | _ => NONE)
  end;

fun default_tycodegen thy defs gr dep thyname brack (TVar ((s, i), _)) =
      SOME (gr, Pretty.str (s ^ (if i = 0 then "" else string_of_int i)))
  | default_tycodegen thy defs gr dep thyname brack (TFree (s, _)) =
      SOME (gr, Pretty.str s)
  | default_tycodegen thy defs gr dep thyname brack (Type (s, Ts)) =
      (case assoc (#types (CodegenData.get thy), s) of
         NONE => NONE
       | SOME (ms, aux) =>
           let
             val (gr', ps) = foldl_map
               (invoke_tycodegen thy defs dep thyname false)
               (gr, fst (args_of ms Ts));
             val (gr'', qs) = foldl_map
               (invoke_tycodegen thy defs dep thyname false)
               (gr', quotes_of ms);
             val thyname' = thyname_of_type s thy;
             val node_id = s ^ " (type)";
             val p = Pretty.block (pretty_mixfix thyname thyname' ms ps qs)
           in SOME (case try (Graph.get_node gr'') node_id of
               NONE => (case get_aux_code aux of
                   [] => (gr'', p)
                 | xs => (Graph.add_edge (node_id, dep) (Graph.new_node
                     (node_id, (NONE, thyname', space_implode "\n" xs ^ "\n")) gr''), p))
             | SOME _ => (Graph.add_edge (node_id, dep) gr'', p))
           end);

val _ = Context.add_setup
 [add_codegen "default" default_codegen,
  add_tycodegen "default" default_tycodegen];


fun mk_struct name s = "structure " ^ name ^ " =\nstruct\n\n" ^ s ^ "end;\n";

fun add_to_module name s ms =
  overwrite (ms, (name, the (assoc (ms, name)) ^ s));

fun output_code gr xs =
  let
    val code =
      map (fn s => (s, Graph.get_node gr s)) (rev (Graph.all_preds gr xs))
    fun string_of_cycle (a :: b :: cs) =
          let val SOME (x, y) = get_first (fn (x, (_, a', _)) =>
            if a = a' then Option.map (pair x)
              (find_first (equal b o #2 o Graph.get_node gr)
                (Graph.imm_succs gr x))
            else NONE) code
          in x ^ " called by " ^ y ^ "\n" ^ string_of_cycle (b :: cs) end
      | string_of_cycle _ = ""
  in
    if "modular" mem !mode then
      let
        val modules = distinct (map (#2 o snd) code);
        val mod_gr = foldr (uncurry Graph.add_edge_acyclic)
          (foldr (uncurry (Graph.new_node o rpair ())) Graph.empty modules)
          (List.concat (map (fn (s, (_, thyname, _)) => map (pair thyname)
            (filter_out (equal thyname) (map (#2 o Graph.get_node gr)
              (Graph.imm_succs gr s)))) code));
        val modules' =
          rev (Graph.all_preds mod_gr (map (#2 o Graph.get_node gr) xs))
      in
        foldl (fn ((_, (_, thyname, s)), ms) => add_to_module thyname s ms)
          (map (rpair "") modules') code
      end handle Graph.CYCLES (cs :: _) =>
        error ("Cyclic dependency of modules:\n" ^ commas cs ^
          "\n" ^ string_of_cycle cs)
    else [("Generated", implode (map (#3 o snd) code))]
  end;

fun gen_generate_code prep_term thy =
  setmp print_mode [] (Pretty.setmp_margin (!margin) (fn xs =>
  let
    val defs = mk_deftab thy;
    val gr = Graph.new_node ("<Top>", (NONE, "Generated", "")) Graph.empty;
    fun expand (t as Abs _) = t
      | expand t = (case fastype_of t of
          Type ("fun", [T, U]) => Abs ("x", T, t $ Bound 0) | _ => t);
    val (gr', ps) = foldl_map (fn (gr, (s, t)) => apsnd (pair s)
      (invoke_codegen thy defs "<Top>" "Generated" false (gr, t)))
        (gr, map (apsnd (expand o prep_term thy)) xs);
    val code =
      space_implode "\n\n" (map (fn (s', p) => Pretty.string_of (Pretty.block
        [Pretty.str ("val " ^ s' ^ " ="), Pretty.brk 1, p, Pretty.str ";"])) ps) ^
      "\n\n"
  in
    map (fn (name, s) => (name, mk_struct name s))
      (add_to_module "Generated" code (output_code gr' ["<Top>"]))
  end));

val generate_code_i = gen_generate_code (K I);
val generate_code = gen_generate_code
  (fn thy => term_of o read_cterm thy o rpair TypeInfer.logicT);


(**** Reflection ****)

val strip_tname = implode o tl o explode;

fun pretty_list xs = Pretty.block (Pretty.str "[" ::
  List.concat (separate [Pretty.str ",", Pretty.brk 1] (map single xs)) @
  [Pretty.str "]"]);

fun mk_type p (TVar ((s, i), _)) = Pretty.str
      (strip_tname s ^ (if i = 0 then "" else string_of_int i) ^ "T")
  | mk_type p (TFree (s, _)) = Pretty.str (strip_tname s ^ "T")
  | mk_type p (Type (s, Ts)) = (if p then parens else I) (Pretty.block
      [Pretty.str "Type", Pretty.brk 1, Pretty.str ("(\"" ^ s ^ "\","),
       Pretty.brk 1, pretty_list (map (mk_type false) Ts), Pretty.str ")"]);

fun mk_term_of thy thyname p (TVar ((s, i), _)) = Pretty.str
      (strip_tname s ^ (if i = 0 then "" else string_of_int i) ^ "F")
  | mk_term_of thy thyname p (TFree (s, _)) = Pretty.str (strip_tname s ^ "F")
  | mk_term_of thy thyname p (Type (s, Ts)) = (if p then parens else I)
      (Pretty.block (separate (Pretty.brk 1)
        (Pretty.str (mk_type_id' (fn s' => "term_of_" ^ s')
          thy thyname (thyname_of_type s thy) s) ::
        List.concat (map (fn T =>
          [mk_term_of thy thyname true T, mk_type true T]) Ts))));


(**** Test data generators ****)

fun mk_gen thy thyname p xs a (TVar ((s, i), _)) = Pretty.str
      (strip_tname s ^ (if i = 0 then "" else string_of_int i) ^ "G")
  | mk_gen thy thyname p xs a (TFree (s, _)) = Pretty.str (strip_tname s ^ "G")
  | mk_gen thy thyname p xs a (Type (s, Ts)) = (if p then parens else I)
      (Pretty.block (separate (Pretty.brk 1)
        (Pretty.str (mk_type_id' (fn s' => "gen_" ^ s')
          thy thyname (thyname_of_type s thy) s ^
          (if s mem xs then "'" else "")) ::
         map (mk_gen thy thyname true xs a) Ts @
         (if s mem xs then [Pretty.str a] else []))));

val test_fn : (int -> (string * term) list option) ref = ref (fn _ => NONE);

fun test_term thy sz i = setmp print_mode [] (fn t =>
  let
    val _ = assert (null (term_tvars t) andalso null (term_tfrees t))
      "Term to be tested contains type variables";
    val _ = assert (null (term_vars t))
      "Term to be tested contains schematic variables";
    val frees = map dest_Free (term_frees t);
    val szname = variant (map fst frees) "i";
    val code = space_implode "\n" (map snd
      (setmp mode ["term_of", "test"] (generate_code_i thy)
        [("testf", list_abs_free (frees, t))]));
    val s = "structure TestTerm =\nstruct\n\n" ^ code ^
      "\nopen Generated;\n\n" ^ Pretty.string_of
        (Pretty.block [Pretty.str "val () = Codegen.test_fn :=",
          Pretty.brk 1, Pretty.str ("(fn " ^ szname ^ " =>"), Pretty.brk 1,
          Pretty.blk (0, [Pretty.str "let", Pretty.brk 1,
            Pretty.blk (0, separate Pretty.fbrk (map (fn (s, T) =>
              Pretty.block [Pretty.str ("val " ^ mk_id s ^ " ="), Pretty.brk 1,
              mk_gen thy "" false [] "" T, Pretty.brk 1,
              Pretty.str (szname ^ ";")]) frees)),
            Pretty.brk 1, Pretty.str "in", Pretty.brk 1,
            Pretty.block [Pretty.str "if ",
              mk_app false (Pretty.str "testf") (map (Pretty.str o mk_id o fst) frees),
              Pretty.brk 1, Pretty.str "then NONE",
              Pretty.brk 1, Pretty.str "else ",
              Pretty.block [Pretty.str "SOME ", Pretty.block (Pretty.str "[" ::
                List.concat (separate [Pretty.str ",", Pretty.brk 1]
                  (map (fn (s, T) => [Pretty.block
                    [Pretty.str ("(" ^ Library.quote (Symbol.escape s) ^ ","), Pretty.brk 1,
                     mk_app false (mk_term_of thy "" false T)
                       [Pretty.str (mk_id s)], Pretty.str ")"]]) frees)) @
                  [Pretty.str "]"])]],
            Pretty.brk 1, Pretty.str "end"]), Pretty.str ");"]) ^
      "\n\nend;\n";
    val _ = use_text Context.ml_output false s;
    fun iter f k = if k > i then NONE
      else (case (f () handle Match =>
          (warning "Exception Match raised in generated code"; NONE)) of
        NONE => iter f (k+1) | SOME x => SOME x);
    fun test k = if k > sz then NONE
      else (priority ("Test data size: " ^ string_of_int k);
        case iter (fn () => !test_fn k) 1 of
          NONE => test (k+1) | SOME x => SOME x);
  in test 0 end);

fun test_goal ({size, iterations, default_type}, tvinsts) i st =
  let
    val thy = Toplevel.theory_of st;
    fun strip (Const ("all", _) $ Abs (_, _, t)) = strip t
      | strip t = t;
    val (gi, frees) = Logic.goal_params
      (prop_of (snd (snd (Proof.get_goal (Toplevel.proof_of st))))) i;
    val gi' = ObjectLogic.atomize_term thy (map_term_types
      (map_type_tfree (fn p as (s, _) => getOpt (assoc (tvinsts, s),
        getOpt (default_type,TFree p)))) (subst_bounds (frees, strip gi)));
  in case test_term (Toplevel.theory_of st) size iterations gi' of
      NONE => writeln "No counterexamples found."
    | SOME cex => writeln ("Counterexample found:\n" ^
        Pretty.string_of (Pretty.chunks (map (fn (s, t) =>
          Pretty.block [Pretty.str (s ^ " ="), Pretty.brk 1,
            Sign.pretty_term thy t]) cex)))
  end;


(**** Interface ****)

val str = setmp print_mode [] Pretty.str;

fun parse_mixfix rd s =
  (case Scan.finite Symbol.stopper (Scan.repeat
     (   $$ "_" >> K Arg
      || $$ "?" >> K Ignore
      || $$ "\\<module>" >> K Module
      || $$ "/" |-- Scan.repeat ($$ " ") >> (Pretty o Pretty.brk o length)
      || $$ "{" |-- $$ "*" |-- Scan.repeat1
           (   $$ "'" |-- Scan.one Symbol.not_eof
            || Scan.unless ($$ "*" -- $$ "}") (Scan.one Symbol.not_eof)) --|
         $$ "*" --| $$ "}" >> (Quote o rd o implode)
      || Scan.repeat1
           (   $$ "'" |-- Scan.one Symbol.not_eof
            || Scan.unless ($$ "_" || $$ "?" || $$ "\\<module>" || $$ "/" || $$ "{" |-- $$ "*")
                 (Scan.one Symbol.not_eof)) >> (Pretty o str o implode)))
       (Symbol.explode s) of
     (p, []) => p
   | _ => error ("Malformed annotation: " ^ quote s));

val _ = Context.add_setup
  [assoc_types [("fun", (parse_mixfix (K dummyT) "(_ ->/ _)",
     [("term_of",
       "fun term_of_fun_type _ T _ U _ = Free (\"<function>\", T --> U);\n"),
      ("test",
       "fun gen_fun_type _ G i =\n\
       \  let\n\
       \    val f = ref (fn x => raise ERROR);\n\
       \    val _ = (f := (fn x =>\n\
       \      let\n\
       \        val y = G i;\n\
       \        val f' = !f\n\
       \      in (f := (fn x' => if x = x' then y else f' x'); y) end))\n\
       \  in (fn x => !f x) end;\n")]))]];


structure P = OuterParse and K = OuterSyntax.Keyword;

fun strip_newlines s = implode (fst (take_suffix (equal "\n")
  (snd (take_prefix (equal "\n") (explode s))))) ^ "\n";

val parse_attach = Scan.repeat (P.$$$ "attach" |--
  Scan.optional (P.$$$ "(" |-- P.xname --| P.$$$ ")") "" --
    (P.verbatim >> strip_newlines));

val assoc_typeP =
  OuterSyntax.command "types_code"
  "associate types with target language types" K.thy_decl
    (Scan.repeat1 (P.xname --| P.$$$ "(" -- P.string --| P.$$$ ")" -- parse_attach) >>
     (fn xs => Toplevel.theory (fn thy => assoc_types
       (map (fn ((name, mfx), aux) => (name, (parse_mixfix
         (typ_of o read_ctyp thy) mfx, aux))) xs) thy)));

val assoc_constP =
  OuterSyntax.command "consts_code"
  "associate constants with target language code" K.thy_decl
    (Scan.repeat1
       (P.xname -- (Scan.option (P.$$$ "::" |-- P.typ)) --|
        P.$$$ "(" -- P.string --| P.$$$ ")" -- parse_attach) >>
     (fn xs => Toplevel.theory (fn thy => assoc_consts
       (map (fn (((name, optype), mfx), aux) => (name, optype, (parse_mixfix
         (term_of o read_cterm thy o rpair TypeInfer.logicT) mfx, aux)))
           xs) thy)));

val generate_codeP =
  OuterSyntax.command "generate_code" "generates code for terms" K.thy_decl
    (Scan.option (P.$$$ "(" |-- P.name --| P.$$$ ")") --
     Scan.optional (P.$$$ "[" |-- P.enum "," P.xname --| P.$$$ "]") (!mode) --
     Scan.repeat1 (P.name --| P.$$$ "=" -- P.term) >>
     (fn ((opt_fname, mode'), xs) => Toplevel.theory (fn thy =>
       let val code = setmp mode mode' (generate_code thy) xs
       in ((case opt_fname of
           NONE => use_text Context.ml_output false
             (space_implode "\n" (map snd code) ^ "\nopen Generated;\n")
         | SOME fname =>
             if "modular" mem mode' then
               app (fn (name, s) => File.write
                   (Path.append (Path.unpack fname) (Path.basic (name ^ ".ML"))) s)
                 (("ROOT", implode (map (fn (name, _) =>
                     "use \"" ^ name ^ ".ML\";\n") code)) :: code)
             else File.write (Path.unpack fname) (snd (hd code))); thy)
       end)));

val params =
  [("size", P.nat >> (K o set_size)),
   ("iterations", P.nat >> (K o set_iterations)),
   ("default_type", P.typ >> set_default_type)];

val parse_test_params = P.short_ident :-- (fn s =>
  P.$$$ "=" |-- getOpt (assoc (params, s), Scan.fail)) >> snd;

fun parse_tyinst xs =
  (P.type_ident --| P.$$$ "=" -- P.typ >> (fn (v, s) => fn thy =>
    fn (x, ys) => (x, (v, typ_of (read_ctyp thy s)) :: ys))) xs;

fun app [] x = x
  | app (f :: fs) x = app fs (f x);

val test_paramsP =
  OuterSyntax.command "quickcheck_params" "set parameters for random testing" K.thy_decl
    (P.$$$ "[" |-- P.list1 parse_test_params --| P.$$$ "]" >>
      (fn fs => Toplevel.theory (fn thy =>
         map_test_params (app (map (fn f => f thy) fs)) thy)));

val testP =
  OuterSyntax.command "quickcheck" "try to find counterexample for subgoal" K.diag
  (Scan.option (P.$$$ "[" |-- P.list1
    (   parse_test_params >> (fn f => fn thy => apfst (f thy))
     || parse_tyinst) --| P.$$$ "]") -- Scan.optional P.nat 1 >>
    (fn (ps, g) => Toplevel.keep (fn st =>
      test_goal (app (getOpt (Option.map
          (map (fn f => f (Toplevel.sign_of st))) ps, []))
        (get_test_params (Toplevel.theory_of st), [])) g st)));

val _ = OuterSyntax.add_keywords ["attach"];

val _ = OuterSyntax.add_parsers
  [assoc_typeP, assoc_constP, generate_codeP, test_paramsP, testP];

end;
