(*  Title:      Pure/Syntax/lexicon.ML
    ID:         $Id$
    Author:     Tobias Nipkow and Markus Wenzel, TU Muenchen

Lexer for the inner Isabelle syntax (terms and types).
*)

signature LEXICON0 =
sig
  val is_identifier: string -> bool
  val is_ascii_identifier: string -> bool
  val is_tid: string -> bool
  val scan_id: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_longid: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_tid: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_nat: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_int: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_hex: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_bin: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_var: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val scan_tvar: SymbolPos.T list -> SymbolPos.T list * SymbolPos.T list
  val implode_xstr: string list -> string
  val explode_xstr: string -> string list
  val read_indexname: string -> indexname
  val read_var: string -> term
  val read_variable: string -> indexname option
  val const: string -> term
  val free: string -> term
  val var: indexname -> term
  val read_nat: string -> int option
  val read_int: string -> int option
  val read_xnum: string -> {radix: int, leading_zeros: int, value: int}
  val fixedN: string
  val constN: string
end;

signature LEXICON =
sig
  include LEXICON0
  val is_xid: string -> bool
  datatype token_kind =
    Literal | IdentSy | LongIdentSy | VarSy | TFreeSy | TVarSy |
    NumSy | XNumSy | StrSy | Space | Comment | EOF
  datatype token = Token of token_kind * string * Position.range
  val str_of_token: token -> string
  val pos_of_token: token -> Position.T
  val is_proper: token -> bool
  val mk_eof: Position.T -> token
  val eof: token
  val is_eof: token -> bool
  val stopper: token Scan.stopper
  val idT: typ
  val longidT: typ
  val varT: typ
  val tidT: typ
  val tvarT: typ
  val terminals: string list
  val is_terminal: string -> bool
  val report_token: token -> unit
  val matching_tokens: token * token -> bool
  val valued_token: token -> bool
  val predef_term: string -> token option
  val tokenize: Scan.lexicon -> bool -> SymbolPos.T list -> token list
end;

structure Lexicon: LEXICON =
struct

(** is_identifier etc. **)

val is_identifier = Symbol.is_ident o Symbol.explode;

fun is_ascii_identifier s =
  let val cs = Symbol.explode s
  in forall Symbol.is_ascii cs andalso Symbol.is_ident cs end;

fun is_xid s =
  (case Symbol.explode s of
    "_" :: cs => Symbol.is_ident cs
  | cs => Symbol.is_ident cs);

fun is_tid s =
  (case Symbol.explode s of
    "'" :: cs => Symbol.is_ident cs
  | _ => false);



(** basic scanners **)

open BasicSymbolPos;

fun !!! msg = SymbolPos.!!! ("Inner lexical error: " ^ msg);

val scan_id = Scan.one (Symbol.is_letter o symbol) ::: Scan.many (Symbol.is_letdig o symbol);
val scan_longid = scan_id @@@ (Scan.repeat1 ($$$ "." @@@ scan_id) >> flat);
val scan_tid = $$$ "'" @@@ scan_id;

val scan_nat = Scan.many1 (Symbol.is_digit o symbol);
val scan_int = $$$ "-" @@@ scan_nat || scan_nat;
val scan_hex = $$$ "0" @@@ $$$ "x" @@@ Scan.many1 (Symbol.is_ascii_hex o symbol);
val scan_bin = $$$ "0" @@@ $$$ "b" @@@ Scan.many1 (fn (s, _) => s = "0" orelse s = "1");

val scan_id_nat = scan_id @@@ Scan.optional ($$$ "." @@@ scan_nat) [];
val scan_var = $$$ "?" @@@ scan_id_nat;
val scan_tvar = $$$ "?" @@@ $$$ "'" @@@ scan_id_nat;



(** datatype token **)

datatype token_kind =
  Literal | IdentSy | LongIdentSy | VarSy | TFreeSy | TVarSy |
  NumSy | XNumSy | StrSy | Space | Comment | EOF;

datatype token = Token of token_kind * string * Position.range;

fun str_of_token (Token (_, s, _)) = s;
fun pos_of_token (Token (_, _, (pos, _))) = pos;

fun is_proper (Token (Space, _, _)) = false
  | is_proper (Token (Comment, _, _)) = false
  | is_proper _ = true;


(* stopper *)

fun mk_eof pos = Token (EOF, "", (pos, Position.none));
val eof = mk_eof Position.none;

fun is_eof (Token (EOF, _, _)) = true
  | is_eof _ = false;

val stopper = Scan.stopper (K eof) is_eof;


(* terminal arguments *)

val idT = Type ("id", []);
val longidT = Type ("longid", []);
val varT = Type ("var", []);
val tidT = Type ("tid", []);
val tvarT = Type ("tvar", []);

val terminals = ["id", "longid", "var", "tid", "tvar", "num", "xnum", "xstr"];
val is_terminal = member (op =) terminals;


(* markup *)

val token_kind_markup =
 fn Literal     => Markup.literal
  | IdentSy     => Markup.ident
  | LongIdentSy => Markup.ident
  | VarSy       => Markup.var
  | TFreeSy     => Markup.tfree
  | TVarSy      => Markup.tvar
  | NumSy       => Markup.num
  | XNumSy      => Markup.xnum
  | StrSy       => Markup.xstr
  | Space       => Markup.none
  | Comment     => Markup.inner_comment
  | EOF         => Markup.none;

fun report_token (Token (kind, _, (pos, _))) =
  Position.report (token_kind_markup kind) pos;


(* matching_tokens *)

fun matching_tokens (Token (Literal, x, _), Token (Literal, y, _)) = x = y
  | matching_tokens (Token (k, _, _), Token (k', _, _)) = k = k';


(* valued_token *)

fun valued_token (Token (Literal, _, _)) = false
  | valued_token (Token (EOF, _, _)) = false
  | valued_token _ = true;


(* predef_term *)

fun predef_term "id" = SOME (Token (IdentSy, "id", Position.no_range))
  | predef_term "longid" = SOME (Token (LongIdentSy, "longid", Position.no_range))
  | predef_term "var" = SOME (Token (VarSy, "var", Position.no_range))
  | predef_term "tid" = SOME (Token (TFreeSy, "tid", Position.no_range))
  | predef_term "tvar" = SOME (Token (TVarSy, "tvar", Position.no_range))
  | predef_term "num" = SOME (Token (NumSy, "num", Position.no_range))
  | predef_term "xnum" = SOME (Token (XNumSy, "xnum", Position.no_range))
  | predef_term "xstr" = SOME (Token (StrSy, "xstr", Position.no_range))
  | predef_term _ = NONE;


(* xstr tokens *)

val scan_chr =
  $$$ "\\" |-- $$$ "'" ||
  Scan.one ((fn s => s <> "\\" andalso s <> "'" andalso Symbol.is_regular s) o symbol) >> single ||
  $$$ "'" --| Scan.ahead (~$$$ "'");

val scan_str =
  $$$ "'" @@@ $$$ "'" @@@ !!! "missing end of string"
    ((Scan.repeat scan_chr >> flat) @@@ $$$ "'" @@@ $$$ "'");

val scan_str_body =
  $$$ "'" |-- $$$ "'" |-- !!! "missing end of string"
    ((Scan.repeat scan_chr >> flat) --| $$$ "'" --| $$$ "'");


fun implode_xstr cs = enclose "''" "''" (implode (map (fn "'" => "\\'" | c => c) cs));

fun explode_xstr str =
  (case Scan.read SymbolPos.stopper scan_str_body (SymbolPos.explode (str, Position.none)) of
    SOME cs => map symbol cs
  | _ => error ("Inner lexical error: literal string expected at " ^ quote str));



(** tokenize **)

fun token_leq (Token (_, s1, _), Token (_, s2, _)) = s1 <= s2;
fun token kind ss = Token (kind, SymbolPos.content ss, SymbolPos.range ss);

fun tokenize lex xids syms =
  let
    val scan_xid =
      if xids then $$$ "_" @@@ scan_id || scan_id
      else scan_id;

    val scan_num = scan_hex || scan_bin || scan_int;

    val scan_val =
      scan_tvar >> token TVarSy ||
      scan_var >> token VarSy ||
      scan_tid >> token TFreeSy ||
      scan_num >> token NumSy ||
      $$$ "#" @@@ scan_num >> token XNumSy ||
      scan_longid >> token LongIdentSy ||
      scan_xid >> token IdentSy;

    val scan_lit = Scan.literal lex >> token Literal;

    val scan_token =
      SymbolPos.scan_comment !!! >> token Comment ||
      Scan.max token_leq scan_lit scan_val ||
      scan_str >> token StrSy ||
      Scan.many1 (Symbol.is_blank o symbol) >> token Space;
  in
    (case Scan.error
        (Scan.finite SymbolPos.stopper (Scan.repeat scan_token)) syms of
      (toks, []) => toks
    | (_, ss) => error ("Inner lexical error at: " ^ SymbolPos.content ss ^
        Position.str_of (#1 (SymbolPos.range ss))))
  end;



(** scan variables **)

(* scan_indexname *)

local

val scan_vname =
  let
    fun nat n [] = n
      | nat n (c :: cs) = nat (n * 10 + (ord c - ord "0")) cs;

    fun idxname cs ds = (implode (rev cs), nat 0 ds);
    fun chop_idx [] ds = idxname [] ds
      | chop_idx (cs as (_ :: "\\<^isub>" :: _)) ds = idxname cs ds
      | chop_idx (cs as (_ :: "\\<^isup>" :: _)) ds = idxname cs ds
      | chop_idx (c :: cs) ds =
          if Symbol.is_digit c then chop_idx cs (c :: ds)
          else idxname (c :: cs) ds;

    val scan = (scan_id >> map symbol) --
      Scan.optional ($$$ "." |-- scan_nat >> (nat 0 o map symbol)) ~1;
  in
    scan >>
      (fn (cs, ~1) => chop_idx (rev cs) []
        | (cs, i) => (implode cs, i))
  end;

in

val scan_indexname = $$$ "'" |-- scan_vname >> (fn (x, i) => ("'" ^ x, i)) || scan_vname;

end;


(* indexname *)

fun read_indexname s =
  (case Scan.read SymbolPos.stopper scan_indexname (SymbolPos.explode (s, Position.none)) of
    SOME xi => xi
  | _ => error ("Lexical error in variable name: " ^ quote s));


(* read_var *)

fun const c = Const (c, dummyT);
fun free x = Free (x, dummyT);
fun var xi = Var (xi, dummyT);

fun read_var str =
  let
    val scan =
      $$$ "?" |-- scan_indexname --| Scan.ahead (Scan.one SymbolPos.is_eof) >> var ||
      Scan.many (Symbol.is_regular o symbol) >> (free o implode o map symbol);
  in the (Scan.read SymbolPos.stopper scan (SymbolPos.explode (str, Position.none))) end;


(* read_variable *)

fun read_variable str =
  let val scan = $$$ "?" |-- scan_indexname || scan_indexname
  in Scan.read SymbolPos.stopper scan (SymbolPos.explode (str, Position.none)) end;


(* specific identifiers *)

val constN = "\\<^const>";
val fixedN = "\\<^fixed>";


(* read numbers *)

local

fun nat cs =
  Option.map (#1 o Library.read_int o map symbol)
    (Scan.read SymbolPos.stopper scan_nat cs);

in

fun read_nat s = nat (SymbolPos.explode (s, Position.none));

fun read_int s =
  (case SymbolPos.explode (s, Position.none) of
    ("-", _) :: cs => Option.map ~ (nat cs)
  | cs => nat cs);

end;


(* read_xnum: hex/bin/decimal *)

local

val ten = ord "0" + 10;
val a = ord "a";
val A = ord "A";
val _ = a > A orelse sys_error "Bad ASCII";

fun remap_hex c =
  let val x = ord c in
    if x >= a then chr (x - a + ten)
    else if x >= A then chr (x - A + ten)
    else c
  end;

fun leading_zeros ["0"] = 0
  | leading_zeros ("0" :: cs) = 1 + leading_zeros cs
  | leading_zeros _ = 0;

in

fun read_xnum str =
  let
    val (sign, radix, digs) =
      (case Symbol.explode (perhaps (try (unprefix "#")) str) of
        "0" :: "x" :: cs => (1, 16, map remap_hex cs)
      | "0" :: "b" :: cs => (1, 2, cs)
      | "-" :: cs => (~1, 10, cs)
      | cs => (1, 10, cs));
    val value = sign * #1 (Library.read_radix_int radix digs);
  in {radix = radix, leading_zeros = leading_zeros digs, value = value} end;

end;

end;
